package com.xebialabs.xlrelease;

import java.io.File;
import java.io.IOException;
import java.lang.annotation.AnnotationFormatError;
import java.nio.charset.StandardCharsets;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.ResultSet;
import java.sql.Statement;
import java.util.*;
import java.util.concurrent.Callable;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.junit.*;
import org.junit.rules.TestName;
import org.junit.runner.RunWith;
import org.junit.runners.JUnit4;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.MDC;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.cache.CacheManager;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.core.io.DefaultResourceLoader;
import org.springframework.core.io.Resource;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.datasource.init.ResourceDatabasePopulator;
import org.springframework.retry.support.RetryTemplate;
import org.springframework.security.authentication.TestingAuthenticationToken;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContext;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.security.util.InMemoryResource;
import org.springframework.test.context.ActiveProfiles;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.junit4.rules.SpringClassRule;
import org.springframework.test.context.junit4.rules.SpringMethodRule;
import org.springframework.test.context.web.WebAppConfiguration;
import org.springframework.transaction.PlatformTransactionManager;
import org.springframework.transaction.TransactionDefinition;
import org.springframework.transaction.support.DefaultTransactionDefinition;
import org.springframework.transaction.support.TransactionTemplate;
import org.springframework.util.StreamUtils;
import org.springframework.util.StringUtils;

import com.xebialabs.deployit.exception.NotFoundException;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plumbing.XLReleaseTest;
import com.xebialabs.deployit.security.PermissionEditor;
import com.xebialabs.deployit.security.PermissionEnforcer;
import com.xebialabs.deployit.security.Role;
import com.xebialabs.deployit.security.RoleService;
import com.xebialabs.deployit.security.permission.Permission;
import com.xebialabs.deployit.security.permission.PermissionHandler;
import com.xebialabs.deployit.util.PasswordEncrypter;
import com.xebialabs.xlrelease.actors.ReleaseActorService;
import com.xebialabs.xlrelease.actors.utils.ReleaseActorLifecycleUtils;
import com.xebialabs.xlrelease.actors.utils.TriggerActorLifecycleUtils;
import com.xebialabs.xlrelease.config.ArchivingSettingsManager;
import com.xebialabs.xlrelease.config.XlrConfig;
import com.xebialabs.xlrelease.db.ArchivedReleases;
import com.xebialabs.xlrelease.domain.*;
import com.xebialabs.xlrelease.domain.events.*;
import com.xebialabs.xlrelease.domain.folder.Folder;
import com.xebialabs.xlrelease.domain.status.ReleaseStatus;
import com.xebialabs.xlrelease.domain.variables.Variable;
import com.xebialabs.xlrelease.events.XLReleaseEventBus;
import com.xebialabs.xlrelease.repository.*;
import com.xebialabs.xlrelease.repository.sql.persistence.FolderPersistence;
import com.xebialabs.xlrelease.security.PermissionChecker;
import com.xebialabs.xlrelease.security.XLReleasePermissions;
import com.xebialabs.xlrelease.security.sql.snapshots.service.PermissionsSnapshotService;
import com.xebialabs.xlrelease.serialization.json.repository.ResolveOptions;
import com.xebialabs.xlrelease.service.*;
import com.xebialabs.xlrelease.spring.configuration.XlrBooterInitializer;
import com.xebialabs.xlrelease.spring.configuration.XlrProfiles;
import com.xebialabs.xlrelease.spring.configuration.XlrWebApplicationInitializer;
import com.xebialabs.xlrelease.utils.Eventually;

import ch.qos.logback.classic.Level;
import ch.qos.logback.classic.spi.ILoggingEvent;
import ch.qos.logback.core.read.ListAppender;
import scala.collection.Seq;
import static scala.jdk.javaapi.CollectionConverters.asScala;
import scala.concurrent.duration.FiniteDuration;

import static com.xebialabs.xlrelease.XLReleaseIntegrationTest.DeleteOption.DELETE_RELEASE;
import static com.xebialabs.xlrelease.XLReleaseIntegrationTest.DeleteOption.DELETE_TEAMS;
import static com.xebialabs.xlrelease.repository.Ids.ROOT_FOLDER_ID;
import static com.xebialabs.xlrelease.rules.LoginRule.grantAdminPermissionTo;
import static com.xebialabs.xlrelease.utils.ConditionBuilder.execute;
import static java.util.Collections.singletonList;
import static java.util.stream.Collectors.joining;
import static java.util.stream.Collectors.toList;

/**
 * This is a base class for JUnit integration tests of XL Release. Inheritors of this class
 * can define {@link Autowired} XL Release services in fields and use them to setup a test
 * environment and assert conditions.
 * <p>
 * A typical test would look like this:
 * </p>
 * <pre>
 *     {@code
 * public class MyIntegrationTest extends XLReleaseIntegrationTest {
 *
 *     &#064;Test
 *     public void should_create_release_in_repository() {
 *         Release release = ReleaseBuilder.newRelease().withId(TestIds.RELEASE1).build();
 *
 *         storeRelease(release);
 *
 *         assertThat(getRelease(TestIds.RELEASE1)).isNotNull();
 *     }
 * }
 * }
 * </pre>
 * <p>
 * <strong>Note:</strong> A test instance of XL Release is setup in a temporary folder and
 * has some services mocked up for speed and easier testing. So you cannot test initializers or
 * upgraders, for example.
 * </p>
 */
@RunWith(JUnit4.class)
@ContextConfiguration(locations = {"/spring/xlrelease-context-test.xml", "classpath:springmvc-resteasy.xml"},
        initializers = {
                XlrWebApplicationInitializer.class,
                XlrBooterInitializer.class,
        })
@WebAppConfiguration("src/test/resources")
@ActiveProfiles(XlrProfiles.INTEGRATION_TEST)
@SuppressWarnings("SpringAutowiredFieldsWarningInspection")
public abstract class XLReleaseIntegrationTest extends XLReleaseTest implements ApplicationContextAware {

    static {
        System.setProperty("config.override_with_env_vars", "true");
        long pid = ProcessHandle.current().pid();
        String userDir = System.getProperty("user.dir");
        boolean userDirIsSameAsWorkdir = userDir.endsWith("build");
        // make sure we always use `build/test-workdir` as a baseDir
        String baseDir = userDirIsSameAsWorkdir ? (userDir + File.separator + "test-workdir" + File.separator + pid) : ("build" + File.separator + "test-workdir" + File.separator + pid);
        System.setProperty("xl.base.dir", baseDir);
        System.setProperty("xl.reporting.engine.location", baseDir + File.separator + "reports");
        System.setProperty("xl.repository.jobLogDir", baseDir + File.separator + "work" + File.separator + "job_logs");
        System.setProperty("xl.repository.workDir", baseDir + File.separator + "work");
        System.setProperty("h2.baseDir", baseDir);
        System.setProperty("logback.baseDir", baseDir);
        // for `derby.system.home` see https://db.apache.org/derby/docs/10.15/ref/rrefproper32066.html
        System.setProperty("derby.system.home", baseDir);
        System.setProperty("derby.drda.startNetworkServer", "true");
    }

    private static final Logger logger = LoggerFactory.getLogger(XLReleaseIntegrationTest.class);

    static {
        String wid = System.getProperty("org.gradle.test.worker");
        if (wid != null) {
            logger.info("Worker id in integration test is: {}", wid);
            // now create isolated schemas for this worker
            String repositoryUrl = System.getProperty("xl.database.db-url", "jdbc:derby:memory:xlrelease;create=true");
            String archiveUrl = System.getProperty("xl.reporting.db-url", "jdbc:derby:memory:xlarchive;create=true");
            String adminUser = System.getProperty("xl.database.adminUser");
            String adminPassword = System.getProperty("xl.database.adminPassword");
            String dbConfig = System.getProperty("xl.repository.configuration");
            String xlreleaseUsername = "xlrelease" + wid;
            String xlarchiveUsername = "xlarchive" + wid;
            if (!dbUserExists(repositoryUrl, xlreleaseUsername, "xlrelease")) {
                Properties connectionProperties = new Properties();
                connectionProperties.setProperty("user", adminUser);
                connectionProperties.setProperty("password", adminPassword);
                try (Connection connection = DriverManager.getConnection(repositoryUrl, connectionProperties)) {
                    ResourceDatabasePopulator scriptPopulator = new ResourceDatabasePopulator();
                    DefaultResourceLoader loader = new DefaultResourceLoader();
                    Resource scriptResource = loader.getResource(dbConfig + "_create_users.sql");
                    String script = StreamUtils.copyToString(scriptResource.getInputStream(), StandardCharsets.UTF_8)
                            .replace("{{xlrelease}}", xlreleaseUsername)
                            .replace("{{xlarchive}}", xlarchiveUsername);
                    scriptPopulator.addScript(new InMemoryResource(script));
                    scriptPopulator.populate(connection);
                } catch (Exception e) {
                    String msg = "Unable to prepare database " + repositoryUrl + " to run integration tests";
                    logger.error(msg, e);
                    throw new IllegalStateException(msg, e);
                }
            }
            // use newly created users
            System.setProperty("xl.database.db-url", dbUrl(repositoryUrl, xlreleaseUsername, "xlrelease"));
            System.setProperty("xl.reporting.db-url", dbUrl(archiveUrl, xlarchiveUsername, "xlarchive"));
            System.setProperty("xl.database.db-username", dbUsername("xlrelease", xlreleaseUsername));
            System.setProperty("xl.reporting.db-username", dbUsername("xlarchive", xlarchiveUsername));
            System.setProperty("xl.database.max-pool-size", "10");
            System.setProperty("xl.reporting.max-pool-size", "10");
            System.setProperty("xl.metrics.enabled", "true");
        }
    }

    private static String dbUrl(String defaultUrl, String newDatabaseName, String databaseName) {
        return defaultUrl.replace(databaseName, newDatabaseName).replace(databaseName.toUpperCase(), newDatabaseName.toUpperCase());
    }

    private static String dbUsername(String defaultUsername, String newUsername) {
        if (System.getProperty("xl.database.db-url").contains("db2")) {
            return defaultUsername;
        } else {
            return newUsername;
        }
    }

    private static boolean dbUserExists(String repositoryUrl, String user, String pass) {
        Properties connectionProperties = new Properties();
        connectionProperties.setProperty("user", dbUsername("xlrelease", user));
        connectionProperties.setProperty("password", pass);
        String newRepositoryUrl = dbUrl(repositoryUrl, user, "xlrelease");
        try (Connection connection = DriverManager.getConnection(newRepositoryUrl, connectionProperties)) {
            // do nothing
            if (repositoryUrl.contains("db2")) {
                try (Statement statement = connection.createStatement()) {
                    String query = "SELECT 1 FROM SYSIBM.SQLSCHEMAS WHERE TABLE_SCHEM like UPPER('" + user + "')";
                    statement.execute(query);
                    try (ResultSet rs = statement.getResultSet()) {
                        boolean res = rs.next();
                        return res;
                    }
                }
            } else {
                return true;
            }
        } catch (Exception ex) {
            logger.error("Unable to check if user {} exists", user, ex);
            return false;
        }
    }

    protected ApplicationContext applicationContext;

    public ApplicationContext getApplicationContext() {
        return applicationContext;
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) {
        this.applicationContext = applicationContext;
    }

    @ClassRule
    public static final SpringClassRule SPRING_CLASS_RULE = new SpringClassRule();

    @Rule
    public final SpringMethodRule springMethodRule = new SpringMethodRule();

    protected boolean featureServicesEnabled() {
        return true;
    }

    protected boolean isRelaxedTest() {
        // relaxed tests rely on integration tests to delete database objects for them
        // we should discourage it since it is a possible cause of flaky tests
        return false;
    }

    protected boolean workDirVerificationEnabled() {
        // check if WorkDir is cleaned or not after test is executed
        // since lot of execution is asynchronous, it causes flaky tests as WorkDirs will be deleted after test cleanup is invoked
        return false;
    }

    public List<String> getCisForDeletion() {
        return cisForDeletion;
    }

    /**
     * Some CIs can not be created inside the @{directory} which is deleted automatically. Those should be added to this collection to be purged after the test.
     */
    protected List<String> cisForDeletion = new ArrayList<>();
    private List<Team> teamsForDeletion = new ArrayList<>();
    private List<String> cisToUnmark = new ArrayList<>();
    private List<AutoCloseable> closeablesToClose = new ArrayList<>();

    @Autowired
    public FolderService folderService;

    @Autowired
    public GateConditionRepository gateConditionRepository;

    @Autowired
    public CommentRepository commentRepository;

    @Autowired
    public ConfigurationService configurationService;

    @Autowired
    public ReleaseService releaseService;

    @Autowired
    public ReleaseRepository releaseRepository;

    @Autowired
    public TriggerRepository triggerRepository;

    @Autowired
    public PhaseRepository phaseRepository;

    @Autowired
    public PlanItemRepository planItemRepository;

    @Autowired
    public UserProfileRepository userProfileRepository;

    @Autowired
    public Eventually eventually;

    @Autowired
    public TeamService teamService;

    @Autowired
    public PermissionsSnapshotService permissionsSnapshotService;

    @Autowired
    public FacetService facetService;

    @Autowired
    public FolderPersistence folderPersistence;

    @Autowired
    public PermissionEditor permissionEditor;

    @Autowired
    private RoleService roleService;

    @Autowired
    protected ReleaseActorLifecycleUtils releaseActorLifecycleUtils;

    @Autowired
    protected TriggerActorLifecycleUtils triggerActorLifecycleUtils;

    @Autowired
    private VariableService variableService;

    @Autowired
    protected ArchivedReleases archivedReleases;

    @Autowired
    protected ArchivingService archivingService;

    @Autowired
    public StorageFacade storageFacade;

    @Autowired
    private TaskRepository taskRepository;

    @Autowired
    protected ConfigurationRepository configurationRepository;

    @Autowired(required = false)
    protected List<FeatureService> featureServices = new ArrayList<>();

    @Qualifier("xlrRepositoryTransactionManager")
    @Autowired
    protected PlatformTransactionManager txManager;

    @Qualifier("reportingJdbcTemplate")
    @Autowired
    protected JdbcTemplate reportingJdbcTemplate;

    @Autowired
    private ArchivingSettingsManager archivingSettings;

    @Autowired
    private ReleaseActorService releaseActorService;

    @Autowired
    private XLReleaseEventBus xlrEventBus;

    @Autowired
    private PreArchiveService preArchiveService;

    @Autowired
    public UserTokenService userTokenService;

    @Autowired
    public UserTokenRepository userTokenRepository;


    @Autowired
    private List<CacheManager> cacheManagers;

    public XLReleaseIntegrationTest() {
    }

    public XLReleaseEventBus eventBus() {
        return xlrEventBus;
    }

    @Rule
    public TestName testName = new TestName();

    @Before
    public void before() {
        String className = getClass().getSimpleName();
        Assert.assertTrue("Integration tests must end with 'IntegrationTest' or 'Documentation'",
                className.endsWith("IntegrationTest") || className.endsWith("Documentation"));
        withAdmin(() -> {
            logger.debug("XLReleaseIntegrationTest 'before' started");
            if (StringUtils.hasText(testName.getMethodName())) {
                MDC.put("testName", testName.getMethodName());
            }

            XLReleasePermissions.init();
            if (featureServicesEnabled()) {
                featureServices.forEach(safeRun(FeatureService::enable));
            } else {
                featureServices.forEach(safeRun(FeatureService::disable));
            }
            archivingSettings.setEnabled(true);
            enablePrearchive();

            grantAdminPermissionTo("admin", permissionEditor, roleService);

            variableService.findGlobalVariablesOrEmpty().getVariables().forEach((ci) -> variableService.deleteGlobalVariable(ci.getId()));

            try {
                folderService.delete("Applications/FolderDefaultReleaseContent");
            } catch (NotFoundException nfe) {
                logger.debug("Can't find default content folder");
            }

            terminateTriggerActors();
            // some actors may be initialized even from non-integration tests, e.g. via ReleaseExecutedEvent, so clean them up
            terminateReleaseActors(FiniteDuration.apply(5, TimeUnit.SECONDS));

            storageFacade.verifyRepositoryClean();
            logger.debug("XLReleaseIntegrationTest 'before' finished");
            return null;
        });
    }

    private Consumer<? super FeatureService> safeRun(Consumer<? super FeatureService> action) {
        return (FeatureService featureService) -> {
            try {
                action.accept(featureService);
            } catch (Exception ex) {
                logger.error("Unable to run feature action on {}", featureService.name(), ex);
            }
        };
    }

    private void disablePrearchive() {
        xlrEventBus.deregister(preArchiveService);
        preArchiveService.disable();
    }

    private void enablePrearchive() {
        preArchiveService.enable();
        xlrEventBus.register(preArchiveService);
    }

    @After
    public void tearDown() throws Exception {
        withAdmin(() -> {
            // 1. disable all services that may run in parallel
            // TODO order feature services
            featureServices.forEach(safeRun(FeatureService::disable));
            disablePrearchive();
            archivingSettings.setEnabled(false);
            waitForEventBusToBecomeEmpty();
            for (AutoCloseable closeable : closeablesToClose) {
                closeable.close();
            }

            try {
                logger.debug("XLReleaseIntegrationTest 'tearDown' started");
                abortRunningReleases();
                terminateTriggerActors();
                cisForDeletion.stream().filter(Ids::isReleaseId).forEach(this::abortReleaseIfNecessary);
                terminateReleaseActors(FiniteDuration.apply(15, TimeUnit.SECONDS));
                deleteCis();
                if (isRelaxedTest()) {
                    storageFacade.cleanup();
                }
                storageFacade.verifyRepositoryClean();

                deleteArchivedReleases();

                if (workDirVerificationEnabled()) {
                    verifyWorkdirClean();
                }

                logger.debug("XLReleaseIntegrationTest 'tearDown' finished");
            } catch (Throwable t) {
                logger.error("Test '{}' tearDown failed.", MDC.get("testName"), t);
                // use AnnotationFormatError since scalatest does not honor other ones
                throw new AnnotationFormatError(t);
            } finally {
                cisForDeletion.clear();
                teamsForDeletion.clear();
                if (StringUtils.hasText(testName.getMethodName())) {
                    MDC.remove("testName");
                }
                try {
                    storageFacade.cleanup();
                } catch (Throwable t) {
                    logger.error("Unable to cleanup database", t);
                    throw new AnnotationFormatError(t);
                }
            }
            return null;
        });
        invalidateCaches();
    }

    private void invalidateCaches() {
        for (CacheManager cm : cacheManagers) {
            for (String cacheName : cm.getCacheNames()) {
                cm.getCache(cacheName).invalidate();
            }
        }
    }

    private void waitForEventBusToBecomeEmpty() {
        logger.info("waiting for event bus to become empty");
        RetryTemplate retry = RetryTemplate.builder().fixedBackoff(50).maxAttempts(15).build();
        retry.execute(context -> {
            if (xlrEventBus.hasPendingMessages()) {
                throw new IllegalStateException("Attempt " + context.getRetryCount());
            } else {
                return null;
            }
        });
        logger.info("event bus considered empty");
    }

    private void terminateTriggerActors() {
        try {
            triggerActorLifecycleUtils.terminateAllTriggerActors();
        } catch (Exception ex) {
            logger.error("Failed while terminating trigger actors", ex);
        }
    }

    private void terminateReleaseActors(final FiniteDuration apply) {
        try {
            releaseActorLifecycleUtils.terminateAllReleaseActorsAndAwait(apply);
        } catch (Exception ex) {
            logger.error("Timed out while terminating actors, try increasing the timeout", ex);
        }
    }

    private void deleteArchivedReleases() {
        reportingJdbcTemplate.execute("DELETE FROM RELEASES");
    }

    protected void verifyRepositoryClean() {
        storageFacade.verifyRepositoryClean();
    }

    protected void verifyWorkdirClean() {
        String workDir = XlrConfig.getInstance().repository().workDir();
        String userDir = System.getProperty("user.dir");
        Path workDirPath = Paths.get(userDir, workDir);
        if (!isDirEmpty(workDirPath)) {
            // clean it up so that the next tests does not fail
            String content = "";
            try {
                content = Files.list(workDirPath).map(Path::toString).collect(joining(",", "\n", ""));
            } catch (IOException e) {
                // ignore
                logger.warn("Unexpected exception", e);
            }
            deleteDirContent(workDirPath);
            throw new TestCleanupException(workDirPath + " was not empty (content will be deleted): " + content);
        }
    }

    private static void deleteDirContent(Path directory) {
        try (DirectoryStream<Path> entries = Files.newDirectoryStream(directory)) {
            for (Path entry : entries) {
                if (entry.toFile().isDirectory()) {
                    deleteDirContent(entry);
                }
                Files.deleteIfExists(entry);
            }
        } catch (IOException e) {
            logger.error("Unexpected exception while deleting content of" + directory, e);
        }
    }

    private static boolean isDirEmpty(final Path directory) {
        try (DirectoryStream<Path> dirStream = Files.newDirectoryStream(directory)) {
            return !dirStream.iterator().hasNext();
        } catch (IOException e) {
            throw new TestCleanupException("Unexpected exception", e);
        }
    }

    private void deleteCis() {
        DefaultTransactionDefinition txDef = new DefaultTransactionDefinition();
        txDef.setName("deleteCisTx");
        txDef.setPropagationBehavior(TransactionDefinition.PROPAGATION_REQUIRES_NEW);
        txDef.setIsolationLevel(TransactionDefinition.ISOLATION_READ_COMMITTED);
        TransactionTemplate template = new TransactionTemplate(txManager, txDef);
        template.execute(transaction -> {
            teamsForDeletion.stream().distinct().collect(toList()).removeIf(t -> storageFacade.delete(t));
            cisForDeletion = cisForDeletion.stream().distinct().collect(toList());
            cisForDeletion.removeAll(cisToUnmark);
            // triggers need to be removed first because they will attempt to lookup templates to unlink
            cisForDeletion.removeIf(id -> Ids.isTriggerId(id) && storageFacade.delete(id));
            cisForDeletion.removeIf(id -> Ids.isDependencyId(id) && storageFacade.delete(id));
            cisForDeletion.removeIf(id -> isPhaseOrTask(id) && storageFacade.delete(id));
            cisForDeletion.removeIf(id -> Ids.isTeamId(id) && storageFacade.delete(id));
            cisForDeletion.removeIf(id -> Ids.isReleaseId(id) && storageFacade.delete(id));
            cisForDeletion.removeIf(id -> Ids.isConfigurationId(id) && storageFacade.delete(id));
            cisForDeletion.removeIf(id -> !Ids.isReleaseId(id) && !Ids.isConfigurationId(id) && !Ids.isFolderId(id) && storageFacade.delete(id));
            cisForDeletion.removeIf(id -> Ids.isFolderId(id) && storageFacade.delete(id));
            cisForDeletion.removeIf(id -> Ids.isFacetId(id) && storageFacade.delete(id));
            return null;
        });
    }

    private void abortRunningReleases() {
        Seq<String> releasesToAbort = releaseRepository.findIdsByKindAndStatus(asScala(List.of(ReleaseKind.RELEASE, ReleaseKind.WORKFLOW)).toSeq(), asScala(Arrays.stream(ReleaseStatus.ACTIVE_STATUSES).collect(Collectors.toList())).toSeq());
        releasesToAbort = releasesToAbort.filter(releaseId -> !abortReleaseIfNecessary(releaseId)).toSeq();
        if (!releasesToAbort.isEmpty()) {
            // use AnnotationFormatError since scalatest does not honor other ones
            throw new AnnotationFormatError("Unable to abort releases, this can cause odd effects in next test, thus crashing here");
        }
    }

    private boolean abortReleaseIfNecessary(final String releaseId) {
        final ReleaseStatus releaseStatus = releaseRepository.getStatus(releaseId);
        if (releaseStatus != null && releaseStatus.isActive()) {
            // abort still running releases
            int abortAttempts = 5;
            long abortBackoff = 1000;
            for (int attempt = 0; attempt < abortAttempts; attempt++) {
                if (attempt > 0) {
                    try {
                        Thread.sleep(abortBackoff);
                        abortBackoff *= 2;
                    } catch (InterruptedException e) {
                        throw new RuntimeException(e);
                    }
                }
                try {
                    releaseActorService.abortRelease(releaseId, "aborted by test tearDown");
                    return true;
                } catch (Exception ex) {
                    logger.warn("Unable to abort release '{}'", releaseId, ex);
                    abortAllTasks(releaseId);
                }
            }
            return false;
        } else {
            return true;
        }
    }

    private void abortAllTasks(final String releaseId) {
        try {
            Release release = releaseService.findById(releaseId);
            release.getAllTasks().stream()
                    .filter(this::abortableTask)
                    .forEach(this::abortTask);
        } catch (Exception ex) {
            logger.warn("Unable to load release {}. Probably it was already deleted.", releaseId, ex);
        }
    }

    private void abortTask(Task t) {
        try {
            String msg = "aborted by tearDown";
            execute(xlrEventBus, () -> releaseActorService.abortTask(t.getId(), msg))
                    .untilAnyOf(
                            new TaskCompletedEvent(t, false),
                            new TaskAbortedEvent(t),
                            new TaskFailedEvent(t, msg),
                            new ReleaseFailedEvent(t.getRelease())
                    );
        } catch (Exception ex) {
            logger.warn("Unable to abort task '{}'", t.getId(), ex);
        }
    }

    private boolean abortableTask(Task task) {
        return task.isInProgress() && task instanceof BaseScriptTask;
    }

    private boolean isPhaseOrTask(final String id) {
        return Ids.isPlanItemId(id) && !Ids.isReleaseId(id);
    }

    protected void deleteOnTearDown(ConfigurationItem... items) {
        for (ConfigurationItem item : items) {
            deleteOnTearDown(item.getId());
        }
    }

    protected void deleteOnTearDown(String... ids) {
        Collections.addAll(cisForDeletion, ids);
    }

    public void markForDeletion(ConfigurationItem item) {
        markForDeletion(item.getId());
    }

    public void markForDeletion(String ciId) {
        logger.info("Marking for deletion: '{}'", ciId);
        cisForDeletion.add(ciId);
    }

    public void markForDeletion(Team team) {
        teamsForDeletion.add(team);
    }

    public void unmarkForDeletion(String id) {
        cisToUnmark.add(id);
        teamsForDeletion.removeIf(team -> id.equals(team.getId()));
    }

    public void deleteOrder(ConfigurationItem... items) {
        cisForDeletion = Stream.concat(Arrays.stream(items).map(ConfigurationItem::getId), cisForDeletion.stream()).collect(toList());
    }

    public void deleteRelease(List<String> releaseIds) {
        releaseIds.forEach(this::deleteRelease);
    }

    public void deleteRelease(String... releaseIds) {
        this.deleteRelease(Arrays.asList(releaseIds));
    }

    public void deleteRelease(String releaseId) {
        releaseRepository.delete(releaseId, false);
    }

    public void storeRelease(Release... releases) {
        storeRelease(Arrays.asList(releases));
    }

    public void storeRelease(List<Release> releases) {
        releases.forEach(this::storeRelease);
    }

    public Release insertArchivedRelease(Release release, String releaseJson, String activityLogs) {
        archivedReleases.insert(release, releaseJson, activityLogs, false);
        markForDeletion(release);
        return release;
    }

    public Release archiveRelease(Release release) {
        storeRelease(release);
        preArchiveAndArchiveRelease(release);
        return release;
    }

    public Release preArchiveAndArchiveRelease(Release release) {
        archivingService.preArchiveRelease(release);
        archivingService.archiveRelease(release.getId());
        return release;
    }

    public int deleteFromArchive(String releaseId) {
        return storageFacade.deleteFromArchive(releaseId);
    }

    public void storeChanges(Changes changes) {
        changes.getUpdatedItems().stream().filter(ci -> ci.getType().equals(Type.valueOf(Release.class)))
                .map(Release.class::cast)
                .forEach(r -> releaseRepository.update(r));
    }

    public <T extends Task> T getTask(String taskId) {
        return taskRepository.findById(taskId, ResolveOptions.WITH_DECORATORS());
    }

    public void createTask(Task task) {
        taskRepository.create(task);
    }

    public void updateTask(Task task) {
        taskRepository.update(task);
    }

    public void updateTaskProperty(Task task) {
        updateTask(task);
    }

    public Phase getPhase(String phaseId) {
        return phaseRepository.findById(phaseId);
    }

    public void updatePhase(Phase phase) {
        phaseRepository.update(phase.getRelease(), phase, phase);
    }

    public void updateRelease(Release release) {
        releaseRepository.update(release);
    }

    /**
     * Safe method to store Release objects including of status TEMPLATE.
     *
     * @param release
     * @return
     * @see #storeTemplate(Release)
     */
    public Release storeRelease(Release release) {
        return this.storeRelease(release, DELETE_RELEASE, DELETE_TEAMS);
    }

    public Trigger storeTrigger(Trigger trigger) {
        Trigger storedTrigger = triggerRepository.create(trigger);
        markForDeletion(storedTrigger);
        return storedTrigger;
    }

    public ReleaseTrigger storeTrigger(ReleaseTrigger trigger) {
        return (ReleaseTrigger) storeTrigger((Trigger) trigger);
    }

    /**
     * Safe method to store Release objects including of status TEMPLATE.
     *
     * @param release
     * @return
     * @see #storeTemplate(Release, DeleteOption...)
     */
    public Release storeRelease(Release release, DeleteOption... deleteOptions) {
        Release stored = releaseRepository.create(release, new CreatedWithoutTemplate());
        logger.info("Stored release '{}' (uid:{}, id: {})", stored.getTitle(), stored.getCiUid(), stored.getId());
        if (optionHas(deleteOptions, DELETE_RELEASE)) {
            markForDeletion(stored);
        }
        storeTeams(stored, deleteOptions);
        // I had a case in Jenkins logs where release was not found by actor immediately after being stored.
        for (int tries = 10; tries > 0; tries--) {
            try {
                Thread.sleep(100);
                releaseRepository.findById(release.getId(), ResolveOptions.WITH_DECORATORS());
                for (Task task : release.getAllTasks()) {
                    taskRepository.findById(task.getId());
                }
                return stored;
            } catch (Exception e) {
                logger.warn("storeRelease check failed: ", e);
            }
        }
        throw new RuntimeException("storeRelease unable to check that release is saved");
    }

    /**
     * Not a safe method to store Release objects of status TEMPLATE - the ID will change.
     *
     * @param releaseData
     * @return
     * @see #storeRelease(Release)
     */
    public Release storeTemplate(Release releaseData) {
        return storeTemplate(releaseData, DELETE_RELEASE, DELETE_TEAMS);
    }

    /**
     * Not a safe method to store Release objects of status TEMPLATE - the ID will change.
     *
     * @param releaseData
     * @return
     * @see #storeRelease(Release, DeleteOption...)
     */
    public Release storeTemplate(Release releaseData, DeleteOption... deleteOptions) {
        return storeTemplate(Folder.ROOT_FOLDER_ID, releaseData, deleteOptions);
    }

    /**
     * Not a safe method to store Release objects of status TEMPLATE - the ID will change.
     *
     * @param releaseData
     * @return
     * @see #storeRelease(Release)
     */
    public Release storeTemplate(Folder parentFolder, Release releaseData) {
        return storeTemplate(parentFolder.getId(), releaseData, DELETE_RELEASE, DELETE_TEAMS);
    }

    /**
     * Not a safe method to store Release objects of status TEMPLATE - the ID will change.
     *
     * @param releaseData
     * @return
     * @see #storeRelease(Release, DeleteOption...)
     */
    public Release storeTemplate(Folder parentFolder, Release releaseData, DeleteOption... deleteOptions) {
        return storeTemplate(parentFolder.getId(), releaseData, deleteOptions);
    }

    private Release storeTemplate(String parentFolderId, Release releaseData, DeleteOption... deleteOptions) {
        Release stored = releaseService.createTemplate(releaseData, parentFolderId);
        if (optionHas(deleteOptions, DELETE_RELEASE)) {
            markForDeletion(stored);
        }
        if (ROOT_FOLDER_ID.equals(parentFolderId)) {
            storeTeams(stored, deleteOptions);
        }
        return stored;
    }

    public <T extends BaseConfiguration> T storeConfiguration(T configurationItem) {
        T storedItem = configurationRepository.create(configurationItem);
        markForDeletion(storedItem);
        return storedItem;
    }

    private void storeTeams(final Release stored, final DeleteOption[] deleteOptions) {
        logger.warn("Saving teams: {}", stored.getTeams().stream().map(Team::getId).collect(joining(",")));
        List<Team> savedTeams = teamService.saveTeamsToPlatform(stored);
        if (optionHas(deleteOptions, DELETE_TEAMS)) {
            savedTeams.forEach(this::markForDeletion);
        }
    }

    private boolean optionHas(final DeleteOption[] deleteOptions, final DeleteOption option) {
        return Arrays.asList(deleteOptions).contains(option);
    }

    public Variable getVariable(String variableId) {
        return variableService.findById(variableId);
    }

    public Release getRelease(String releaseId) {
        return releaseRepository.findById(releaseId);
    }

    public GateCondition getGateCondition(String conditionId) {
        return gateConditionRepository.findById(conditionId);
    }

    public PlanItem getPlanItem(String planItemId) {
        return planItemRepository.findById(planItemId);
    }

    public String createRole(String roleName) {
        return createRole(roleName, Collections.emptyList());
    }

    public String createRole(String roleName, List<String> principals) {
        Role role = new Role(roleName);
        role.setPrincipals(principals);
        List<Role> allRoles = roleService.readRoleAssignments();
        allRoles.add(role);
        roleService.writeRoleAssignments(allRoles);
        return roleName;
    }

    public Folder createFolder(Folder folder, boolean createDefaultTeamsIfTopLevel) {
        return createFolder(Ids.getParentId(folder.getId()), folder, createDefaultTeamsIfTopLevel);
    }

    public Folder createFolder(String parentId, Folder folder) {
        return createFolder(parentId, folder, true);
    }

    public Folder createFolder(String parentId, Folder folder, boolean createDefaultTeamsIfTopLevel) {
        Folder createdFolder = folderService.create(parentId, folder, createDefaultTeamsIfTopLevel);
        markForDeletion(createdFolder);
        return createdFolder;
    }

    public String encrypt(final String password) {
        if (null != password && !XlrConfig.getInstance().repository_decryptPasswords()) {
            return PasswordEncrypter.getInstance().ensureEncrypted(password);
        } else {
            return password;
        }
    }

    public enum DeleteOption {
        DELETE_NONE,
        DELETE_RELEASE,
        DELETE_TEAMS
    }

    public <T extends AutoCloseable> T registerCloseable(T closeable) {
        closeablesToClose.add(closeable);
        return closeable;
    }

    protected void withAdmin(Callable<Void> callable) {
        SecurityContext context = SecurityContextHolder.getContext();
        Authentication oldAuth = context.getAuthentication();
        Authentication adminAuth = new TestingAuthenticationToken("itusername", "itpassword", PermissionEnforcer.ROLE_ADMIN);
        context.setAuthentication(adminAuth);
        SecurityContextHolder.setContext(context);
        try {
            callable.call();
        } catch (Exception ex) {
            // use AnnotationFormatError since scalatest does not honor other exceptions
            throw new AnnotationFormatError(ex);
        } finally {
            context.setAuthentication(oldAuth);
        }
    }

    public List<ILoggingEvent> captureLogs(String... names) {
        ListAppender<ILoggingEvent> listAppender = new ListAppender<>();
        Map<ch.qos.logback.classic.Logger, Level> oldLevels = new HashMap<>();
        for (String name : names) {
            ch.qos.logback.classic.Logger captureLogger = (ch.qos.logback.classic.Logger) LoggerFactory.getLogger(name);
            captureLogger.addAppender(listAppender);
            captureLogger.setLevel(Level.DEBUG);
        }
        listAppender.start();
        registerCloseable(makeLogAppenderCloser(listAppender, oldLevels));
        return listAppender.list;
    }

    private AutoCloseable makeLogAppenderCloser(ListAppender<ILoggingEvent> appender, Map<ch.qos.logback.classic.Logger, Level> oldLevels) {
        return new AutoCloseable() {
            @Override
            public void close() throws Exception {
                oldLevels.entrySet().forEach(entry -> {
                    entry.getKey().setLevel(entry.getValue());
                    entry.getKey().detachAppender(appender);
                });
                appender.stop();
                appender.list.clear();
            }
        };
    }

    public void grantGlobalPermissions(String username, Set<Permission> permissions) {
        permissions.forEach(permission -> {
            assert permission.getLevel() == PermissionHandler.Level.GLOBAL || permission.getLevel() == PermissionHandler.Level.BOTH;
        });
        String testName = MDC.get("testName");
        if (null == testName) testName = "";
        Role roleWithPermission = new Role(String.valueOf(roleService.getRoles().size()), "TestRole_" + testName + "_" + username);
        roleWithPermission.getPrincipals().add(username);
        Map<Role, Set<Permission>> rolePermission = new HashMap<>();
        rolePermission.put(roleWithPermission, permissions);
        roleService.writeRoleAssignments(singletonList(roleWithPermission));
        permissionEditor.editPermissions(PermissionChecker.GLOBAL_SECURITY_ALIAS(), rolePermission);
    }
}
