package com.xebialabs.deployit.plumbing.serialization

import com.xebialabs.deployit.plugin.api.services.Repository
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.xlrelease.domain.folder.Folder
import com.xebialabs.xlrelease.repository.ConfigurationRepository
import com.xebialabs.xltype.serialization.{CiReference, ConfigurationItemConverter}
import grizzled.slf4j.Logging

import scala.collection.mutable.ListBuffer
import scala.jdk.CollectionConverters._
import scala.util.{Failure, Success, Try}

trait ConfigurationItemReferenceResolver extends ResolutionContextAware with Logging {
  this: ConfigurationItemConverter =>
  def getConfigurationRepository: ConfigurationRepository

  def resolveReferencesWithFallback(repository: Repository): Unit = {
    getReferences.forEach(reference => {
      // Resolve IDs to CIs
      val resolvedCis = ListBuffer.empty[ConfigurationItem]

      reference.getIds.forEach(id => {
        if (id != null) {
          val alsoRead = getReadCIs.get(id)

          if (alsoRead != null) {
            resolvedCis += alsoRead
          } else {
            // Keep for backwards compatibility
            Try(repository.read[ConfigurationItem](id)) match {
              case Success(ci) =>
                if (ci != null) {
                  resolvedCis += ci
                } else {
                  resolvePortableReferenceId(reference, id).foreach(resolvedCis.+=)
                }
              case Failure(ex) =>
                logger.trace(s"Unable to directly resolve reference ${reference}", ex)
                resolvePortableReferenceId(reference, id).foreach(resolvedCis.+=)
            }
          }
        }
      })

      // Set referred CIs on the parsed CI
      reference.set(resolvedCis.asJava)
    })
  }

  private def resolvePortableReferenceId(reference: CiReference, id: String): Option[ConfigurationItem] = {
    val ciType = reference.getProperty.getReferencedType
    PortableConfigurationReference.from(ciType, id).flatMap(reference => {
      logger.trace(s"Loading Custom Configuration '$id' by name")
      reference.resolve(getConfigurationRepository, getResolutionContext)
    })
  }
}

trait ResolutionContextAware {
  def getResolutionContext: ResolutionContext
}

class ResolutionContext private(val folderId: Option[String])

object ResolutionContext {
  def apply(folderId: Option[String]): ResolutionContext = new ResolutionContext(folderId.filterNot(_ == Folder.ROOT_FOLDER_ID))

  def apply(folderId: String): ResolutionContext = apply(Option(folderId))

  lazy val GLOBAL = new ResolutionContext(None)
}
