package com.xebialabs.xlrelease.api.v1.impl;

import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import com.codahale.metrics.annotation.Timed;

import com.xebialabs.deployit.checks.Checks;
import com.xebialabs.xlrelease.actors.ReleaseActorService;
import com.xebialabs.xlrelease.api.v1.PhaseApi;
import com.xebialabs.xlrelease.domain.Phase;
import com.xebialabs.xlrelease.domain.Release;
import com.xebialabs.xlrelease.domain.Task;
import com.xebialabs.xlrelease.repository.PhaseVersion;
import com.xebialabs.xlrelease.security.PermissionChecker;
import com.xebialabs.xlrelease.service.PhaseService;
import com.xebialabs.xlrelease.service.ReleaseService;

import static com.google.common.base.Strings.isNullOrEmpty;
import static com.xebialabs.deployit.checks.Checks.checkArgument;
import static com.xebialabs.xlrelease.repository.Ids.releaseIdFrom;

@Controller
public class PhaseApiImpl implements PhaseApi {

    private static final Logger logger = LoggerFactory.getLogger(PhaseApiImpl.class);

    private final PermissionChecker permissions;
    private final PhaseService phaseService;
    private final ReleaseService releaseService;
    private final ReleaseActorService releaseActorService;

    @Autowired
    public PhaseApiImpl(PermissionChecker permissions, PhaseService phaseService, ReleaseService releaseService,
                    ReleaseActorService releaseActorService) {
        this.permissions = permissions;
        this.phaseService = phaseService;
        this.releaseService = releaseService;
        this.releaseActorService = releaseActorService;
    }

    @Timed
    @Override
    public Phase getPhase(String phaseId) {
        permissions.checkView(releaseIdFrom(phaseId));
        return phaseService.findById(phaseId);
    }

    @Timed
    @Override
    public Phase updatePhase(String phaseId, Phase phase) {
        permissions.checkEdit(releaseIdFrom(phaseId));
        return releaseActorService.updatePhase(phaseId, phase);
    }

    @Timed
    @Override
    public Phase updatePhase(Phase phase) {
        return updatePhase(phase.getId(), phase);
    }

    @Timed
    @Override
    public Task addTask(String containerId, Task task, Integer position) {
        String releaseId = releaseIdFrom(containerId);
        permissions.checkView(releaseId);
        permissions.checkEdit(releaseId);
        permissions.checkEditTask(releaseId);
        if (task.isLocked()) {
            permissions.checkLockTaskPermission(releaseId);
        }
        if (task.getPrecondition() != null) {
            permissions.checkEditPreconditionPermission(releaseId);
        }
        if (task.getFailureHandler() != null || task.isTaskFailureHandlerEnabled() || task.getTaskRecoverOp() != null) {
            permissions.checkEditFailureHandlerPermission(releaseId);
        }

        try {
            return releaseActorService.createTask(containerId, task, position);
        } catch (IllegalArgumentException e) {
            logger.error("Could not add a task to a phase", e);
            throw new Checks.IncorrectArgumentException(e.getMessage());
        }
    }

    @Timed
    @Override
    public List<Phase> searchPhasesByTitle(String phaseTitle, String releaseId) {
        checkArgument(!isNullOrEmpty(phaseTitle), "Query parameter phaseTitle must be provided");
        checkArgument(!isNullOrEmpty(releaseId), "Query parameter releaseId must be provided");
        Release release = releaseService.findById(releaseId);
        permissions.checkView(release);

        return release.getPhasesByTitle(phaseTitle);
    }

    @Timed
    @Override
    public List<Phase> searchPhases(String phaseTitle, String releaseId, PhaseVersion phaseVersion) {
        checkArgument(!isNullOrEmpty(releaseId), "Query parameter releaseId must be provided");
        Release release = releaseService.findById(releaseId);
        permissions.checkView(release);

        List<Phase> filteredPhases = release.getPhasesContainingInTitle(phaseTitle);

        filteredPhases = filterPhasesForPhaseVersion(filteredPhases, phaseVersion); // filter by phaseVersion (if provided)

        Collections.reverse(filteredPhases);
        return filteredPhases;
    }

    @Timed
    @Override
    public Phase addPhase(String releaseId, Phase phase, Integer position) {
        permissions.checkView(releaseId);
        permissions.checkEdit(releaseId);
        try {
            return releaseActorService.addPhase(releaseId, phase, position);
        } catch (IllegalArgumentException e) {
            throw new Checks.IncorrectArgumentException(e.getMessage());
        }
    }

    @Timed
    @Override
    public Phase copyPhase(String phaseIdToCopy, int targetPosition) {
        String releaseId = releaseIdFrom(phaseIdToCopy);
        permissions.copyPhase(releaseId);
        try {
            return releaseActorService.copyPhase(releaseId, phaseIdToCopy, targetPosition);
        } catch (IllegalArgumentException e) {
            throw new Checks.IncorrectArgumentException(e.getMessage());
        }
    }

    @Timed
    @Override
    public Phase addPhase(String releaseId, Phase phase) {
        return addPhase(releaseId, phase, null);
    }

    @Timed
    @Override
    public Phase newPhase() {
        return newPhase("New Phase");
    }

    @Timed
    @Override
    public Phase newPhase(String title) {
        Phase phase = new Phase();
        phase.setTitle(title);
        return phase;
    }

    @Timed
    @Override
    public void deletePhase(String phaseId) {
        permissions.checkEdit(releaseIdFrom(phaseId));
        releaseActorService.deletePhase(phaseId);
    }

    private List<Phase> filterPhasesForPhaseVersion(List<Phase> phases, final PhaseVersion phaseVersion) {
        if (phaseVersion == null || phaseVersion == PhaseVersion.ALL) return phases;

        return phases.stream()
                .filter(phase -> PhaseVersion.isPhaseVersion(phase, phaseVersion))
                .collect(Collectors.toList());
    }
}
