package com.xebialabs.xlrelease.export;

import java.io.IOException;
import java.io.OutputStream;
import java.io.StringWriter;
import java.lang.annotation.Annotation;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;
import javax.ws.rs.core.MediaType;
import org.jboss.resteasy.core.Headers;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import com.xebialabs.deployit.plumbing.CurrentVersion;
import com.xebialabs.deployit.plumbing.export.ImportTemplateConfigurationItemReaderWriter;
import com.xebialabs.xlplatform.upgrade.RepositoryVersionService;
import com.xebialabs.xlrelease.domain.Release;
import com.xebialabs.xlrelease.utils.CiHelper;
import com.xebialabs.xltype.serialization.json.JsonWriter;

import static com.google.common.collect.Lists.newArrayList;
import static com.xebialabs.xlrelease.export.TemplateExportHelper.addAttachments;
import static com.xebialabs.xlrelease.upgrade.Components.XL_RELEASE_COMPONENT;
import static com.xebialabs.xlrelease.utils.CiHelper.getNestedCis;
import static java.nio.charset.StandardCharsets.UTF_8;

@Service
public class TemplateExporter {
    public static final String RELEASE_TEMPLATE_JSON_FILE = "release-template.json";
    public static final String MANIFEST_JSON_FILE = "manifest.json";
    public static final String XLR_VERSION_FIELD_NAME = "xlr-version";
    public static final String DATA_MODEL_VERSION_FIELD_NAME = "xlr-data-model-version";

    private ImportTemplateConfigurationItemReaderWriter configurationItemReaderWriter;
    private RepositoryVersionService repositoryVersionService;

    @Autowired
    public TemplateExporter(ImportTemplateConfigurationItemReaderWriter configurationItemReaderWriter, RepositoryVersionService repositoryVersionService) {
        this.configurationItemReaderWriter = configurationItemReaderWriter;
        this.repositoryVersionService = repositoryVersionService;
    }

    public void exportTemplate(Release template, OutputStream outputStream) {
        getNestedCis(newArrayList(template)).forEach(CiHelper::eraseTokens);

        try (ZipOutputStream zipOutputStream = new ZipOutputStream(outputStream)) {
            addAttachments(template, zipOutputStream);
            addReleaseTemplate(template, zipOutputStream);
            addManifestFile(zipOutputStream);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private void addReleaseTemplate(Release template, ZipOutputStream zipOutputStream) throws IOException {
        zipOutputStream.putNextEntry(new ZipEntry(RELEASE_TEMPLATE_JSON_FILE));

        // Ensure the phases/tasks/sub-tasks are saved as well.
        configurationItemReaderWriter.writeTo(template, Release.class, Release.class, new Annotation[]{}, MediaType.APPLICATION_JSON_TYPE, new Headers<>(), zipOutputStream);

        zipOutputStream.closeEntry();
    }

    private void addManifestFile(ZipOutputStream zipOutputStream) throws IOException {
        zipOutputStream.putNextEntry(new ZipEntry(MANIFEST_JSON_FILE));

        StringWriter stringWriter = new StringWriter();
        JsonWriter writer = new JsonWriter(stringWriter);
        writer.object();
        writer.key(DATA_MODEL_VERSION_FIELD_NAME).value(repositoryVersionService.readVersionOfComponent(XL_RELEASE_COMPONENT).getVersion());
        writer.key(XLR_VERSION_FIELD_NAME).value(CurrentVersion.get());
        writer.endObject();

        String formattedJson = stringWriter.toString();
        zipOutputStream.write(formattedJson.getBytes(UTF_8));

        zipOutputStream.closeEntry();
    }

}
