package com.xebialabs.xlrelease.scheduler.upgrade

import com.xebialabs.deployit.server.api.upgrade.{Upgrade, Version}
import com.xebialabs.xlrelease.domain._
import com.xebialabs.xlrelease.domain.status.TaskStatus
import com.xebialabs.xlrelease.repository.Ids
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.{TASKS, TASK_JOBS}
import com.xebialabs.xlrelease.scheduler._
import com.xebialabs.xlrelease.scheduler.repository.JobRepository
import com.xebialabs.xlrelease.script.TaskSoftReference
import com.xebialabs.xlrelease.service.ReleaseService
import com.xebialabs.xlrelease.upgrade.Components.XL_RELEASE_COMPONENT
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper}

import java.sql.ResultSet
import scala.collection.mutable
import scala.concurrent.duration.Duration
import scala.jdk.CollectionConverters._
import scala.util.Try

class RecoveryActorUpgrade @Autowired()(@Qualifier("xlrRepositoryJdbcTemplate") val jdbcTemplate: JdbcTemplate,
                                        val releaseService: ReleaseService,
                                        val jobRepository: JobRepository,
                                     ) extends Upgrade with Logging{

  override def upgradeVersion(): Version = Version.valueOf(XL_RELEASE_COMPONENT, "10.3.0#2")

  override def doUpgrade(): Boolean = {
    logger.info("Finding tasks, that should be executed, but do not have relevant jobs in the queue")

    val taskIdsToCheckByReleaseId: Map[String, mutable.Buffer[String]] = jdbcTemplate.query(
      s"""
         |SELECT ${TASKS.TASK_ID}
         |FROM ${TASKS.TABLE}
         |WHERE ${TASKS.STATUS} = '${TaskStatus.IN_PROGRESS.value()}'
         |   OR ${TASKS.STATUS} = '${TaskStatus.ABORT_SCRIPT_IN_PROGRESS.value()}'
         |   OR ${TASKS.STATUS} = '${TaskStatus.FAILURE_HANDLER_IN_PROGRESS.value()}'
         |   OR ${TASKS.STATUS} = '${TaskStatus.FACET_CHECK_IN_PROGRESS.value()}'
         |   OR ${TASKS.STATUS} = '${TaskStatus.PRECONDITION_IN_PROGRESS.value()}'
         |""".stripMargin,
      new RowMapper[String] {
        override def mapRow(rs: ResultSet, rowNum: Int): String = rs.getString(1)
      }).asScala.groupBy(Ids.releaseIdFrom)

    val taskIdsThatHaveJobs: Set[String] = jdbcTemplate.query(
      s"""
         |SELECT ${TASK_JOBS.TASK_ID}
         |FROM ${TASK_JOBS.TABLE}
         |""".stripMargin,
      new RowMapper[String] {
        override def mapRow(rs: ResultSet, rowNum: Int): String = Ids.getFolderlessId(rs.getString(1))
      }).asScala.toSet

    taskIdsToCheckByReleaseId.foreach {
      case (releaseId, taskIds) => Try(upgradeRelease(releaseId, taskIds.toSet,taskIdsThatHaveJobs))
        .recover(e => {
          logger.error(s"Unable to process release ${releaseId}, giving up", e)
        })
    }


    logger.info("Done")
    true
  }

  //noinspection ScalaStyle
  private def upgradeRelease(releaseId: String, taskIds: Set[String], taskIdsThatHaveJobs: Set[String]): Unit = {
    val release = releaseService.findById(releaseId)
    val tasksToCheck = release.getAllTasks.asScala.filter(task => {
      val taskFolderlessId = Ids.getFolderlessId(task.getId)
      taskIds.contains(taskFolderlessId) &&
        !taskIdsThatHaveJobs.contains(taskFolderlessId)
    })
      .toSeq
    tasksToCheck.foreach(task => {
      Try({
        if (resumeOnRecovery(task)) {
          // it is only CustomScriptTask that hasNextScriptToExecute can be recovered.
          // We are leaving task in IN_PROGRESS, and the job, that we create here will be
          // eventually picked by JobCleanupService and queued for execution
          jobRepository.create(JobRow(NextCustomScriptTaskJob(taskRef(task.asInstanceOf[CustomScriptTask]), Duration.Zero)))
          logger.info(s"Task ${task.getId} will be recovered")
        } else if (failOnRecovery(task)) {
          // we can't silently change status of this task to FAILED, because this will break release
          // execution flow and will not send notifications to user(s), but
          // since actor system is not ready yet, we can't use ReleaseActorService to fail task
          // as well.
          // So, we'll create job in the table, that will look like task was actually started (reserved
          // state), and it will be eventually failed by JobCleanupService
          logger.info(s"Task ${task.getId} will be failed")
          for {
            taskJob <- task.getStatus match {
              case TaskStatus.PRECONDITION_IN_PROGRESS => Some(PreconditionJob(taskRef(task)))
              case TaskStatus.FAILURE_HANDLER_IN_PROGRESS => Some(FailureHandlerJob(taskRef(task)))
              case TaskStatus.FACET_CHECK_IN_PROGRESS => Some(FacetCheckJob(taskRef(task)))
              case TaskStatus.ABORT_SCRIPT_IN_PROGRESS => Some(CustomScriptTaskJob(taskRef(task.asInstanceOf[CustomScriptTask])))
              case TaskStatus.IN_PROGRESS => task match {
                case customScriptTask: CustomScriptTask => Some(CustomScriptTaskJob(taskRef(customScriptTask)))
                case scriptTask: ResolvableScriptTask => Some(ScriptTaskJob(taskRef(scriptTask)))
                case createReleaseTask: CreateReleaseTask => Some(CreateReleaseTaskJob(taskRef(createReleaseTask)))
                case notificationTask: NotificationTask => Some(NotificationTaskJob(taskRef(notificationTask)))
                case _ => None
              }
              case _ => None
            }
            jobRow = JobRow(taskJob).copy(status = JobStatus.RUNNING)
          } jobRepository.create(jobRow)
        }
      }).recover(e => {
        logger.error(s"Unable to process task ${task.getId}, giving up", e)
      })
    })
  }

  private def resumeOnRecovery(task: Task): Boolean = task match {
    case customScriptTask: CustomScriptTask if customScriptTask.hasNextScriptToExecute || customScriptTask.isWaitingForSignal => true
    case _ => false
  }

  private def failOnRecovery(task: Task): Boolean = task match {
    case _ if task.isFailureHandlerInProgress || task.isFacetInProgress || task.isPreconditionInProgress => true
    case taskGroup if task.isInstanceOf[TaskGroup] => false
    case _ => ((task.isAutomated && (task.isInProgress || task.isAbortScriptInProgress)))
  }

  private def taskRef[T <: Task](task: T): TaskSoftReference[T] = new TaskSoftReference[T](task, () => task)

}
