package com.xebialabs.xlrelease.security.filter;

import java.io.IOException;
import javax.servlet.FilterChain;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.springframework.security.authentication.AuthenticationDetailsSource;
import org.springframework.security.authentication.AuthenticationManager;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.AuthenticationException;
import org.springframework.security.core.context.SecurityContext;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.security.web.AuthenticationEntryPoint;
import org.springframework.security.web.authentication.WebAuthenticationDetailsSource;
import org.springframework.util.Assert;
import org.springframework.web.filter.OncePerRequestFilter;

import com.xebialabs.xlrelease.authentication.InvalidTokenException;
import com.xebialabs.xlrelease.security.authentication.RunnerAuthenticationToken;
import com.xebialabs.xlrelease.utils.TokenTypes;

/**
 * filter to get personal access token from request header or basic auth header
 */
public class RunnerAuthenticationFilter extends OncePerRequestFilter {

    private final AuthenticationEntryPoint authenticationEntryPoint;

    private final AuthenticationManager authenticationManager;

    private final String header;

    private final AuthenticationDetailsSource<HttpServletRequest, ?> authenticationDetailsSource = new WebAuthenticationDetailsSource();

    public RunnerAuthenticationFilter(AuthenticationManager authenticationManager,
                                      AuthenticationEntryPoint authenticationEntryPoint,
                                      String header) {
        Assert.notNull(authenticationManager, "authenticationManager cannot be null");
        Assert.notNull(authenticationEntryPoint, "authenticationEntryPoint cannot be null");
        Assert.hasText(header, "header cannot be empty");
        this.authenticationManager = authenticationManager;
        this.authenticationEntryPoint = authenticationEntryPoint;
        this.header = header;
    }

    @Override
    protected void doFilterInternal(final HttpServletRequest request, final HttpServletResponse response, final FilterChain filterChain) throws ServletException, IOException {
        String token = this.getTokenFromHeader(request);
        if (token == null) {
            logger.trace("Did not process request since did not find bearer token");
            filterChain.doFilter(request, response);
            return;
        }
        if (!isTokenValid(token)) {
            logger.trace("Did not process request since supplied token is not a valid runner authentication token");
            throw new InvalidTokenException("Supplied token is not a valid runner authentication token");
        }

        try {
            RunnerAuthenticationToken authenticationToken = new RunnerAuthenticationToken(token);
            authenticationToken.setDetails(this.authenticationDetailsSource.buildDetails(request));

            Authentication authenticationResult = this.authenticationManager.authenticate(authenticationToken);
            SecurityContext context = SecurityContextHolder.createEmptyContext();
            context.setAuthentication(authenticationResult);
            SecurityContextHolder.setContext(context);
            logger.debug(String.format("Set SecurityContextHolder to %s", authenticationResult));
            filterChain.doFilter(request, response);
        } catch (AuthenticationException ex) {
            SecurityContextHolder.clearContext();
            logger.trace("Failed to process authentication request", ex);
            this.authenticationEntryPoint.commence(request, response, ex);
        }
    }

    private String getTokenFromHeader(HttpServletRequest request) {
        return request.getHeader(header);
    }

    private boolean isTokenValid(String token) {
        return token.startsWith(TokenTypes.RRA().toString());
    }

}
