package com.xebialabs.xlrelease.service;

import java.io.InputStream;
import java.util.Set;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import com.codahale.metrics.annotation.Timed;

import com.xebialabs.deployit.exception.NotFoundException;
import com.xebialabs.deployit.io.StreamWrappingOverthereFile;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.xlrelease.config.XlrConfig;
import com.xebialabs.xlrelease.domain.Attachment;
import com.xebialabs.xlrelease.domain.Release;
import com.xebialabs.xlrelease.domain.Task;
import com.xebialabs.xlrelease.domain.events.AttachmentCreatedEvent;
import com.xebialabs.xlrelease.domain.events.AttachmentDeletedEvent;
import com.xebialabs.xlrelease.events.XLReleaseEventBus;
import com.xebialabs.xlrelease.exception.LogFriendlyNotFoundException;
import com.xebialabs.xlrelease.repository.AttachmentRepository;
import com.xebialabs.xlrelease.repository.AttachmentSizeLimiter;
import com.xebialabs.xlrelease.repository.ReleaseRepository;

import static com.google.common.base.Preconditions.checkState;
import static java.lang.String.format;

@Service
public class AttachmentService {

    private static final Logger logger = LoggerFactory.getLogger(AttachmentService.class);

    private AttachmentRepository attachmentRepository;
    private ReleaseRepository releaseRepository;
    private ArchivingService archivingService;
    private CiIdService ciIdService;
    private XLReleaseEventBus eventBus;

    @Autowired
    public AttachmentService(AttachmentRepository attachmentRepository,
                             ReleaseRepository releaseRepository,
                             ArchivingService archivingService,
                             CiIdService ciIdService,
                             XLReleaseEventBus eventBus) {
        this.attachmentRepository = attachmentRepository;
        this.releaseRepository = releaseRepository;
        this.archivingService = archivingService;
        this.ciIdService = ciIdService;
        this.eventBus = eventBus;
    }

    @Timed
    public Attachment findById(String attachmentId) {
        return attachmentRepository.findById(attachmentId);
    }

    @Timed
    public Attachment findByIdIncludingArchived(String attachmentId) {
        try {
            return findById(attachmentId);
        } catch (NotFoundException e) {
            if (archivingService.attachmentExists(attachmentId)) {
                return archivingService.getAttachment(attachmentId);
            } else {
                throw new LogFriendlyNotFoundException(format("Attachment [%s] does not exist in the repository or archive", attachmentId), e);
            }
        }
    }

    public void createAttachmentOnReleaseFromActor(Release release, Attachment attachment) {
        attachmentRepository.create(release, attachment);
        eventBus.publish(new AttachmentCreatedEvent(release.getId(), attachment));
    }

    public void createAttachmentOnTaskFromActor(Release release, String taskId, Attachment attachment) {
        attachmentRepository.create(release, taskId, attachment);
        eventBus.publish(new AttachmentCreatedEvent(taskId, attachment));
    }

    public Attachment attachToRelease(Release release, String filename, String contentType, InputStream bytes) {
        AttachmentSizeLimiter limiter = new AttachmentSizeLimiter(XlrConfig.getInstance().server().upload().maxSize(), bytes);
        StreamWrappingOverthereFile file = new StreamWrappingOverthereFile(filename, limiter);
        Attachment attachment = new Attachment(file, contentType);
        String attachmentId = ciIdService.getUniqueId(Type.valueOf(Attachment.class), release.getId());
        attachment.setId(attachmentId);
        attachment.getExportFilename();
        release.getAttachments().add(attachment);
        return attachment;
    }

    @Timed
    public void deleteAttachment(String releaseId, String attachmentId) {
        Release release = findReleaseById(releaseId);
        Set<Task> tasksUsingAttachment = release.getTasksUsingAttachment(attachmentId);
        checkState(tasksUsingAttachment.isEmpty(), "Attachment '%s' is used by tasks : %s", attachmentId, tasksUsingAttachment);

        Attachment attachment = findById(attachmentId);

        attachmentRepository.delete(release, attachment);
        eventBus.publish(new AttachmentDeletedEvent(releaseId, attachment));
    }

    @Timed
    public void deleteAttachmentFromTask(String releaseId, String taskId, String attachmentId) {
        Release release = findReleaseById(releaseId);
        Task task = release.getTask(taskId);

        LockedTaskOperationChecks.checkDeleteAttachment(task);

        checkState(!task.isInProgress(), "Can't delete an attachment from a running task.");
        Set<Task> tasksUsingAttachment = release.getTasksUsingAttachment(attachmentId);
        checkState(tasksUsingAttachment.contains(task), "Attachment '%s' found on release, but not found on task '%s'", attachmentId, taskId);

        Attachment attachment = task.getAttachments().stream()
                .filter(at -> attachmentId.equals(at.getId()))
                .findFirst()
                .orElseThrow(() -> new LogFriendlyNotFoundException("Attachment [%s] not found", attachmentId));

        attachmentRepository.delete(task, attachment);
        eventBus.publish(new AttachmentDeletedEvent(taskId, attachment));
    }

    private Release findReleaseById(String releaseId) {
        return releaseRepository.findById(releaseId);
    }

    public String insertArtifact(Release release, String artifactName, InputStream content) {
        String artifactId = ciIdService.getUniqueId(Type.valueOf(Attachment.class), release.getId());
        AttachmentSizeLimiter limitedContent = new AttachmentSizeLimiter(XlrConfig.getInstance().server().upload().maxSize(), content);
        attachmentRepository.insert(release.getCiUid(), artifactId, artifactName, limitedContent);
        return artifactId;
    }
}