package com.xebialabs.xlrelease.service;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;
import javax.activation.FileTypeMap;
import org.apache.commons.fileupload.FileItemIterator;
import org.apache.commons.fileupload.FileItemStream;
import org.apache.commons.fileupload.FileUploadException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import com.codahale.metrics.annotation.Timed;

import com.xebialabs.xlrelease.actors.ReleaseActorService;
import com.xebialabs.xlrelease.domain.Attachment;
import com.xebialabs.xlrelease.domain.Release;
import com.xebialabs.xlrelease.domain.Task;
import com.xebialabs.xlrelease.repository.AttachmentRepository;
import com.xebialabs.xlrelease.repository.Ids;
import com.xebialabs.xlrelease.repository.ReleaseRepository;
import com.xebialabs.xlrelease.utils.FileContentValidation;

@Service
public class UploadService {

    private static final Logger logger = LoggerFactory.getLogger(UploadService.class);

    private final AttachmentService attachmentService;
    private final ReleaseRepository releaseRepository;
    private final AttachmentRepository attachmentRepository;
    private final ReleaseActorService releaseActorService;

    @Autowired
    public UploadService(AttachmentService attachmentService,
                         ReleaseRepository releaseRepository,
                         AttachmentRepository attachmentRepository,
                         ReleaseActorService releaseActorService) {
        this.attachmentService = attachmentService;
        this.releaseRepository = releaseRepository;
        this.attachmentRepository = attachmentRepository;
        this.releaseActorService = releaseActorService;
    }

    @Timed
    public List<Attachment> addAttachment(String ciId, FileItemIterator fileItems) throws IOException, FileUploadException {
        List<Attachment> attachmentList = new ArrayList<>();

        while (fileItems.hasNext()) {
            FileItemStream item = fileItems.next();
            if (item.isFormField()) {
                continue;
            }
            String filename = item.getName();
            String contentType = item.getContentType();
            try (InputStream stream = item.openStream();
                 InputStream validatedInputStream = FileContentValidation.apply(filename, stream).get()
            ) {
                logger.debug("Attaching {} to {}", filename, ciId);
                attachmentList.add(addAttachment(ciId, filename, contentType, validatedInputStream));
            }
        }

        return attachmentList;
    }

    @Timed
    public Attachment addAttachment(String ciId, String fileName, byte[] fileByteArray) {
        return addAttachment(ciId, fileName, FileTypeMap.getDefaultFileTypeMap().getContentType(fileName), new ByteArrayInputStream(fileByteArray));
    }

    @Timed
    public Attachment addAttachment(String ciId, String fileName, String contentType, InputStream inputStream) {
        logger.debug("Attaching {} to {}", fileName, ciId);
        Release release = releaseRepository.findById(Ids.releaseIdFrom(ciId));
        Task task = null;
        if (!Ids.isReleaseId(ciId)) {
            task = release.getTask(ciId);
        }

        Attachment attachment = attachmentService.attachToRelease(release, fileName, contentType, inputStream);
        if (task != null) {
            task.getAttachments().add(attachment);
        }
        attachmentRepository.insertAttachment(release.getId(), attachment);
        if (null != task) {
            return releaseActorService.createAttachmentOnTask(task.getId(), attachment);
        } else {
            return releaseActorService.createAttachmentOnRelease(release.getId(), attachment);
        }
    }
}
