package com.xebialabs.xlrelease.spring.configuration

import com.codahale.metrics.InstrumentedScheduledExecutorService
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.metrics.XlrMetricRegistry
import com.xebialabs.xlrelease.utils.PrefixedThreadFactory
import org.springframework.context.annotation.{Bean, Configuration}
import org.springframework.scheduling.concurrent.ConcurrentTaskScheduler

import java.util.concurrent.{ScheduledExecutorService, ScheduledThreadPoolExecutor}


object SpringTaskSchedulerConfiguration {
  final val SPRING_SCHEDULED_EXECUTOR_SERVICE = "springScheduledExecutorService"
  final val TASK_SCHEDULER = "taskScheduler"
  final val SPRING_SCHEDULED_TASK = "spring-scheduled-task"
}

/**
 * Default Task scheduler used by spring if {@link org.springframework.scheduling.annotation.EnableScheduling} annotation is present
 */
@Configuration
class SpringTaskSchedulerConfiguration(val xlrConfig: XlrConfig) {

  @Bean(name = Array(SpringTaskSchedulerConfiguration.SPRING_SCHEDULED_EXECUTOR_SERVICE), destroyMethod = "shutdown")
  def springScheduledExecutorService: ScheduledExecutorService = {
    val scheduledExecutorService = new ScheduledThreadPoolExecutor(xlrConfig.executors.springScheduler.maxThreadsCount,
      new PrefixedThreadFactory(SpringTaskSchedulerConfiguration.SPRING_SCHEDULED_TASK, true))

    if (xlrConfig.metrics.enabled) {
      new InstrumentedScheduledExecutorService(scheduledExecutorService, XlrMetricRegistry.metricRegistry, SpringTaskSchedulerConfiguration.SPRING_SCHEDULED_TASK)
    } else {
      scheduledExecutorService
    }
  }

  @Bean(name = Array(SpringTaskSchedulerConfiguration.TASK_SCHEDULER))
  def taskScheduler = new ConcurrentTaskScheduler(springScheduledExecutorService)
}
