package com.xebialabs.xlrelease.upgrade

import grizzled.slf4j.Logging
import org.springframework.transaction.support.TransactionTemplate

object UpgradeSupport {

  trait BatchSupport {
    self: Logging =>
    val BATCH_SIZE = 20

    case class Batch[T](items: Iterable[T], group: Int, logPrefix: String)

    def doInBatch[T](items: Iterable[T], batchSize: Int = BATCH_SIZE, itemsName: String = "items")(block: Batch[T] => Unit): Unit = {
      val totalItemsNum = items.size
      val totalBatchNum = Math.max(1, (totalItemsNum.toDouble / batchSize.toDouble).ceil.toInt)
      val batchDigits = digits(totalBatchNum)

      def padBatchNum: Int => String = pad(batchDigits)

      def batchCounter(batchNum: Int): String = s"Batch[${padBatchNum(batchNum)}/${padBatchNum(totalBatchNum)}]"

      items.grouped(batchSize).zipWithIndex.foreach { case (batchItems, batchGroup) =>
        val logPrefix = batchCounter(batchGroup + 1)
        logger.info(s"$logPrefix: Upgrading ${batchItems.size} $itemsName")
        block(Batch(batchItems, batchGroup, logPrefix))
      }
    }

    protected def digits(num: Int): Int = Math.max(1, Math.log10(num.toDouble).ceil.toInt)

    protected def pad(size: Int, leadingZeros: Boolean = true)(num: Int): String =
      s"%${if (leadingZeros) "0" else ""}${Math.max(1, size)}d".format(num)
  }

  trait TransactionSupport {
    self: Logging =>
    def transactionTemplate: TransactionTemplate

    def doInTransaction(block: => Unit): Unit = {
      transactionTemplate.execute(_ =>
        try {
          block
        } catch {
          case e: Exception =>
            val errorMsg = s"Unable to successfully finish database transaction"
            logger.error(errorMsg, e)
            throw new IllegalStateException(errorMsg, e)
        }
      )
    }
  }

}
