package com.xebialabs.xlrelease.api.internal

import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

import java.util.{Collection => JCollection, List => JList}
import scala.jdk.CollectionConverters._

@Component
class InternalMetadataDecoratorService extends Logging {

  private var decorators: Seq[InternalMetadataDecorator[_]] = Seq()

  @Autowired(required = false)
  def setDecorators(decorators: JList[_ <: InternalMetadataDecorator[_]]): Unit = {
    this.decorators = decorators.asScala.toSeq
  }

  def decorate(ci: ConfigurationItem, decoratorNames: JList[String]): Unit = {
    decorate(ci, decoratorNames, DecoratorsCache.NO_CACHE)
  }

  def decorate(cis: JCollection[_ <: ConfigurationItem], decoratorNames: JList[String]): Unit = {
    decorate(cis, decoratorNames, DecoratorsCache.NO_CACHE)
  }

  def decorate(ci: ConfigurationItem, decoratorNames: JList[String], cache: DecoratorsCache): Unit = {
    getApplicableDecorators(ci, decoratorNames).map(_.asInstanceOf[InternalMetadataDecorator[ConfigurationItem]]).foreach { decorator =>
      logger.trace(s"Applying decorator ${decorator.name} on CI ${ci.getId}")
      decorator.decorateWithCache(ci, decorator.name, cache)
    }
  }

  def decorate(cis: JCollection[_ <: ConfigurationItem], decoratorNames: JList[String], cache: DecoratorsCache): Unit = {
    getApplicableDecorators(cis.asScala, decoratorNames).foreach { decorator =>
      logger.trace(s"Applying decorator ${decorator.name} on ${cis.size()} CIs")
      decorator.asInstanceOf[InternalMetadataDecorator[ConfigurationItem]].decorateWithCache(cis.asInstanceOf[JCollection[ConfigurationItem]], decorator.name, cache)
    }
  }

  private def getApplicableDecorators(cis: Iterable[_ <: ConfigurationItem], names: JList[String]): Seq[InternalMetadataDecorator[_]] = {
    cis.headOption.map(getApplicableDecorators(_, names)).getOrElse(Seq())
  }

  private def getApplicableDecorators(ci: ConfigurationItem, names: JList[String]): Seq[InternalMetadataDecorator[_]] = {
    decorators.filter { decorator =>
      names.contains(decorator.name) && decorator.isApplicableTo(ci)
    }
  }

}
