package com.xebialabs.xlrelease.service

import com.codahale.metrics.annotation.Timed
import com.xebialabs.xlrelease.config.ArchivingSettingsManager
import com.xebialabs.xlrelease.configuration.ArchivingSettings.DEFAULT_SEARCH_PAGE_SIZE
import com.xebialabs.xlrelease.repository.ReleaseRepository
import grizzled.slf4j.Logging
import org.joda.time.LocalDateTime
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service

import java.util.Date
import java.util.concurrent.atomic.{AtomicBoolean, AtomicInteger}
import javax.annotation.PreDestroy
import scala.util.{Failure, Try}

@Service
class ArchivingScheduleService @Autowired()(val archivingService: ArchivingService,
                                            val archivingConfig: ArchivingSettingsManager,
                                            val releaseSearchService: ReleaseSearchService,
                                            val releaseRepository: ReleaseRepository) extends Logging {

  private val isCancelRequested = new AtomicBoolean(false)
  private val archiveInProgressLock = new Object

  def getArchivingJobCronSchedule: String = archivingConfig.getArchivingJobCronSchedule

  @Timed
  def processExpiredReleases(releaseAgeToArchive: Integer): Unit = {
    if (archivingConfig.getEnabled) {
      val now = LocalDateTime.now()
      val expirationDate = now.minusHours(releaseAgeToArchive).toDate
      val expirationDateWithGrace = now.minusHours(releaseAgeToArchive).minusMinutes(15).toDate // 15 minutes grace for pre-archival

      logger.debug(s"Processing completed releases older than $releaseAgeToArchive hours (completed before $expirationDate)...")

      archiveInProgressLock.synchronized {
        val runStartTimeMillis = System.currentTimeMillis()

        val archivedReleasesCount = findAndProcessArchivableReleases(expirationDate, runStartTimeMillis, archivingService.findArchivableReleaseIds) {
          releaseId =>
            logger.info(s"Archiving release [$releaseId]")
            archiveCompletedRelease(releaseId)
        }

        var deletedReleaseCount = 0;
        if (!ranTooLong(runStartTimeMillis)) {
          deletedReleaseCount = findAndProcessArchivableReleases(expirationDateWithGrace, runStartTimeMillis, releaseSearchService.findNonArchivedExpiredReleaseIds) {
            releaseId =>
              logger.info(s"Deleting release [$releaseId] marked do not archive")
              archiveCompletedRelease(releaseId)
          }
        }

        logger.debug(s"Archived $archivedReleasesCount and deleted $deletedReleaseCount releases in ${System.currentTimeMillis() - runStartTimeMillis} ms")
      }
    } else {
      logger.warn("Archiving is not enabled.")
    }
  }

  @Timed
  def processExpiredReleases(): Unit =
    processExpiredReleases(archivingConfig.getReleaseAgeToDeleteFromJcr)

  private def findAndProcessArchivableReleases(date: Date, runStartTimeMillis: Long, getReleases: (Date, Int) => Seq[String])
                                              (action: String => Try[Unit]): Int = {
    val count = new AtomicInteger(0)
    var done = false

    while (!done && !isCancelRequested.get()) {
      val archivingPageSize: Int = getSearchPageSize

      logger.debug(s"Fetching $archivingPageSize completed or aborted releases from repository")
      val releaseIds = getReleases(date, archivingPageSize)

      if (releaseIds.nonEmpty && count.get() == 0) {
        logger.info(s"Archiving releases that completed before $date")
      }

      done = processReleases(releaseIds, count, runStartTimeMillis)(action)

      if (!done) {
        if (releaseIds.isEmpty || releaseIds.size < archivingPageSize) {
          logger.debug(s"Archived only ${releaseIds.size} finished releases on this page: we're done.")
          done = true
        } else {
          logger.debug(s"Archived ${count.get()} releases that completed before $date, fetching the next $archivingPageSize")
        }
      }
    }

    if (count.get() > 0) {
      logger.info(s"Archived ${count.get()} releases that completed before $date")
    }
    count.get()
  }

  //noinspection ScalaStyle
  private def processReleases(releaseIds: Iterable[String], count: AtomicInteger, runStartTimeMillis: Long)
                             (action: String => Try[Unit]): Boolean = {
    for (releaseId <- releaseIds) {
      if (isCancelRequested.get()) {
        return true
      }

      sleepIfNeeded()

      action(releaseId).foreach(_ => count.incrementAndGet())

      if (ranTooLong(runStartTimeMillis)) {
        logger.warn(s"Archiving paused because it ran for more than ${archivingConfig.getMaxSecondsPerRun} seconds. " +
          s"Archiving will continue in the next run")
        return true
      }
    }

    false
  }

  private def archiveCompletedRelease(releaseId: String): Try[Unit] = {
    Try {
      val s = System.currentTimeMillis()
      archivingService.archiveRelease(releaseId)
      logger.debug(s"Archived release [$releaseId] in ${System.currentTimeMillis() - s} ms")
    }.recoverWith {
      case e: Exception =>
        logger.error(s"Could not archive release [$releaseId]: ${e.getMessage}", e)
        Failure(e)
    }
  }

  private def getSearchPageSize: Int =
    if (archivingConfig.getSearchPageSize > 0) archivingConfig.getSearchPageSize else DEFAULT_SEARCH_PAGE_SIZE

  private def sleepIfNeeded(): Unit = {
    if (archivingConfig.getSleepSecondsBetweenReleases > 0) {
      logger.debug(s"Sleeping for ${archivingConfig.getSleepSecondsBetweenReleases} seconds before archiving the next release")
      try {
        Thread.sleep(archivingConfig.getSleepSecondsBetweenReleases * 1000)
      } catch {
        case _: InterruptedException =>
          logger.warn(s"Sleeping for ${archivingConfig.getSleepSecondsBetweenReleases} seconds was interrupted")
      }
    }
  }

  private def ranTooLong(runStartTimeMillis: Long): Boolean = {
    if (archivingConfig.getMaxSecondsPerRun > 0) {
      (System.currentTimeMillis() - runStartTimeMillis) > (archivingConfig.getMaxSecondsPerRun * 1000)
    } else {
      false
    }
  }

  private def off(): Unit = {
    isCancelRequested.set(true)
    logger.debug("Waiting until no archiving is in progress")
    archiveInProgressLock.synchronized {
      logger.debug("Archiving done, destroying ArchivingScheduleService")
    }
  }

  @PreDestroy
  private def preDestroy(): Unit = {
    off()
  }
}
