package com.xebialabs.deployit.plumbing.authentication;

import java.io.IOException;
import java.io.InputStream;
import java.util.Map;
import javax.mail.internet.ContentType;
import javax.mail.internet.ParseException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.security.authentication.AuthenticationServiceException;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.AuthenticationException;
import org.springframework.security.web.authentication.AbstractAuthenticationProcessingFilter;
import org.springframework.security.web.util.matcher.AntPathRequestMatcher;
import com.fasterxml.jackson.core.JsonParseException;
import com.fasterxml.jackson.databind.JsonMappingException;
import com.fasterxml.jackson.databind.ObjectMapper;

import static org.springframework.security.web.authentication.UsernamePasswordAuthenticationFilter.SPRING_SECURITY_FORM_PASSWORD_KEY;
import static org.springframework.security.web.authentication.UsernamePasswordAuthenticationFilter.SPRING_SECURITY_FORM_USERNAME_KEY;

public class UsernamePasswordAuthenticationFilter extends AbstractAuthenticationProcessingFilter {
    private static final Logger logger = LoggerFactory.getLogger(UsernamePasswordAuthenticationFilter.class);

    private static final ObjectMapper objectMapper = new ObjectMapper();    // it's thread safe

    public UsernamePasswordAuthenticationFilter() {
        super(new AntPathRequestMatcher("/login", "POST"));
    }

    public Authentication attemptAuthentication(HttpServletRequest request,
                                                HttpServletResponse response) throws AuthenticationException {
        if (!"POST".equals(request.getMethod())) {
            throw new AuthenticationServiceException("Authentication method must be a POST got: " + request.getMethod());
        }

        String contentType = request.getContentType();
        if (request.getContentType() == null) {
            throw new AuthenticationServiceException("Authentication payload no content type set (expecting application/json)");
        }
        try {
            ContentType type = new ContentType(contentType);
            if (type.match("application/json")) {
                return getAuthenticationJson(request);
            } else if (type.match("application/x-www-form-urlencoded")) {
                return getAuthenticationFormEncoded(request);
            } else {
                // 406 return would be nicer, but requires more code
               throw new AuthenticationServiceException("Content type must be either `application/json` or `application/x-www-form-urlencoded` got: " + contentType);
            }
        } catch (ParseException e) {
            throw new AuthenticationServiceException("Problem parsing content type: " + contentType, e);
        }
    }

    private Authentication getAuthenticationJson(final HttpServletRequest request) {
        try {
            InputStream is = request.getInputStream();
            Map<String, String> data = (Map<String, String>) objectMapper.readValue(is, Map.class);

            String username = data.getOrDefault(SPRING_SECURITY_FORM_USERNAME_KEY, "");
            username = username.trim();

            String password = data.getOrDefault(SPRING_SECURITY_FORM_PASSWORD_KEY, "");

            return getAuthentication(request, username, password);
        } catch (JsonParseException e) {
            logger.error("Problem decoding username/password", e);
            throw new AuthenticationServiceException("Could not parse authentication payload", e);
        } catch (JsonMappingException e) {
            logger.error("Problem decoding username/password", e);
            throw new AuthenticationServiceException("Could not parse authentication payload", e);
        } catch (IOException e) {
            // IO Exceptions are normal in a networked world, prevent spamming log with errors that just may happen
            logger.debug("I/O problem decoding username/password {}", e.getMessage());
            throw new AuthenticationServiceException("I/O problem decoding username/password", e);
        }
    }

    private Authentication getAuthenticationFormEncoded(final HttpServletRequest request) {
        try {
            String username = getRequestParamOrEmpty(request,  SPRING_SECURITY_FORM_USERNAME_KEY).trim();
            String password = getRequestParamOrEmpty(request, SPRING_SECURITY_FORM_PASSWORD_KEY);
            return getAuthentication(request, username, password);
        } catch(Exception e) {
            logger.error("problem reading username and/or password {}", e.getMessage());
            throw new AuthenticationServiceException("problem reading username and/or password.", e);
        }
    }

    private String getRequestParamOrEmpty(final HttpServletRequest request, String parameter) {
        String parameterValue = request.getParameter(parameter);
        if (parameterValue != null) {
            return parameterValue;
        } else {
            return "";
        }
    }

    private Authentication getAuthentication(final HttpServletRequest request, final String username, final String password) {
        UsernamePasswordAuthenticationToken authRequest = new UsernamePasswordAuthenticationToken(username, password);

        authRequest.setDetails(authenticationDetailsSource.buildDetails(request));
        return this.getAuthenticationManager().authenticate(authRequest);
    }

}
