package com.xebialabs.xlrelease.config;

import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.xlrelease.configuration.ArchivingSettings;
import com.xebialabs.xlrelease.repository.ConfigurationRepository;

import static com.xebialabs.xlrelease.configuration.ArchivingSettings.ARCHIVING_SETTINGS_ID;

/**
 * A wrapper around {@link com.xebialabs.xlrelease.configuration.ArchivingSettings} which manages
 * both SQL-stored settings and JMX-controlled settings.
 */
@Component
public class ArchivingSettingsManager {

    private static final Logger logger = LoggerFactory.getLogger(ArchivingSettingsManager.class);

    private ConfigurationRepository configurationRepository;

    private Boolean preArchivingEnabled;
    private Boolean enabled;
    private Integer searchPageSize;
    private Integer maxSecondsPerRun;
    private Integer sleepSecondsBetweenReleases;
    private Integer purgingSearchPageSize;
    private Integer purgingMaxSecondsPerRun;
    private Integer purgingSleepSecondsBetweenReleases;

    private Consumer<Boolean> preArchivingEnabledChangedCallback;

    @Autowired
    public ArchivingSettingsManager(ConfigurationRepository configurationRepository) {
        this.configurationRepository = configurationRepository;
        initializeDefaults();
    }

    private void initializeDefaults() {
        ArchivingSettings defaults = Type.valueOf(ArchivingSettings.class).getDescriptor().newInstance(ARCHIVING_SETTINGS_ID);
        this.preArchivingEnabled = defaults.getPreArchivingEnabled();
        this.enabled = defaults.getEnabled();
        this.searchPageSize = defaults.getSearchPageSize();
        this.maxSecondsPerRun = defaults.getMaxSecondsPerRun();
        this.sleepSecondsBetweenReleases = defaults.getSleepSecondsBetweenReleases();
        this.purgingMaxSecondsPerRun = defaults.getPurgingMaxSecondsPerRun();
        this.purgingSearchPageSize = defaults.getPurgingSearchPageSize();
        this.purgingSleepSecondsBetweenReleases = defaults.getPurgingSleepSecondsBetweenReleases();
    }

    public void subscribeToPreArchivingEnabledChanges(Consumer<Boolean> callback) {
        this.preArchivingEnabledChangedCallback = callback;
    }

    public boolean getPreArchivingEnabled() {
        return preArchivingEnabled;
    }

    public void setPreArchivingEnabled(Boolean enabled) {
        logger.info("Setting pre-archiving enabled to be " + enabled);
        if (preArchivingEnabledChangedCallback != null && enabled != this.preArchivingEnabled) {
            preArchivingEnabledChangedCallback.accept(enabled);
        }
        this.preArchivingEnabled = enabled;
    }

    public Integer getReleaseAgeToDeleteFromJcr() {
        return getStoredArchivingSettings().getReleaseAgeToDeleteFromJcr();
    }

    public String getArchivingJobCronSchedule() {
        return getStoredArchivingSettings().getArchivingJobCronSchedule();
    }

    public boolean getEnabled() {
        return enabled;
    }

    public void setEnabled(final Boolean enabled) {
        logger.info("Setting archiving enabled to be " + enabled);
        this.enabled = enabled;
    }

    public Integer getSearchPageSize() {
        return searchPageSize;
    }

    public void setSearchPageSize(final Integer searchPageSize) {
        logger.info("Setting archiving search page size to be " + searchPageSize);
        this.searchPageSize = searchPageSize;
    }

    public Integer getMaxSecondsPerRun() {
        return maxSecondsPerRun;
    }

    public void setMaxSecondsPerRun(final Integer maxSecondsPerRun) {
        logger.info("Setting archiving max seconds per run to be " + maxSecondsPerRun);
        this.maxSecondsPerRun = maxSecondsPerRun;
    }

    public Integer getSleepSecondsBetweenReleases() {
        return sleepSecondsBetweenReleases;
    }

    public void setSleepSecondsBetweenReleases(final Integer sleepSecondsBetweenReleases) {
        logger.info("Setting archiving sleep seconds between releases to be " + sleepSecondsBetweenReleases);
        this.sleepSecondsBetweenReleases = sleepSecondsBetweenReleases;
    }

    public Integer getMaximumArchiveRetentionPeriod() {
        return getStoredArchivingSettings().getMaximumArchiveRetentionPeriod();
    }

    public String getPurgingJobCronSchedule() {
        return getStoredArchivingSettings().getPurgingJobCronSchedule();
    }

    public boolean getPurgingEnabled() {
        return getStoredArchivingSettings().getPurgingEnabled();
    }

    public String getPurgeExportPath() {
        return getStoredArchivingSettings().getPurgeExportPath();
    }

    public boolean getExportOnPurgeEnabled() {
        return getStoredArchivingSettings().getExportOnPurgeEnabled();
    }

    public Integer getPurgingSearchPageSize() {
        return purgingSearchPageSize;
    }

    public void setPurgingSearchPageSize(final Integer searchPageSize) {
        logger.info("Setting purging search page size to be " + searchPageSize);
        this.purgingSearchPageSize = searchPageSize;
    }

    public Integer getPurgingMaxSecondsPerRun() {
        return purgingMaxSecondsPerRun;
    }

    public void setPurgingMaxSecondsPerRun(final Integer maxSecondsPerRun) {
        logger.info("Setting purging max seconds per run to be " + maxSecondsPerRun);
        this.purgingMaxSecondsPerRun = maxSecondsPerRun;
    }

    public Integer getPurgingSleepSecondsBetweenReleases() {
        return purgingSleepSecondsBetweenReleases;
    }

    public void setPurgingSleepSecondsBetweenReleases(final Integer sleepSecondsBetweenReleases) {
        logger.info("Setting purging sleep seconds between releases to be " + sleepSecondsBetweenReleases);
        this.purgingSleepSecondsBetweenReleases = sleepSecondsBetweenReleases;
    }

    private ArchivingSettings getStoredArchivingSettings() {
        return configurationRepository.read(ARCHIVING_SETTINGS_ID);
    }
}
