package com.xebialabs.xlrelease.security.authentication.policy

import com.xebialabs.deployit.plumbing.authentication.InternalUser
import com.xebialabs.xlrelease.domain.UserProfile
import com.xebialabs.xlrelease.service.{UserProfileService, Users}
import grizzled.slf4j.Logging
import org.springframework.security.authentication.{BadCredentialsException, RememberMeAuthenticationToken}
import org.springframework.security.core.Authentication
import org.springframework.stereotype.Component

@Component
class DefaultUserProfileCreationPolicy(userProfileService: UserProfileService,
                                       usersRepository: Users)
  extends UserProfileCreationPolicy with Logging {

  override def order(): Int = 1000 // Is the default catch-all policy, should be last to try out

  override def policyApplies(authentication: Authentication): Boolean = true // catch-all

  override def createProfile(authentication: Authentication): UserProfile = {
    // if authentication is of an external user (e.g. LDAP, Crowd), then we need to make sure, that
    // internal user with same username does not exist
    if (!authentication.isInstanceOf[RememberMeAuthenticationToken]) {
      authentication.getPrincipal match {
        case _: InternalUser => ()
        case _ if usersRepository.userExistsInRepository(authentication.getName) =>
          logger.warn(s"Unable to create profile for user [${authentication.getName}]. A local account exists.")
          throw new BadCredentialsException(s"A local account exists for user ${authentication.getName}")
        case _ => ()
      }
    }
    userProfileService.createOrUpdate(authentication.getName)
  }

}
