package com.xebialabs.xlrelease.service

import com.codahale.metrics.annotation.Timed
import com.xebialabs.xlrelease.config.ArchivingSettingsManager
import com.xebialabs.xlrelease.configuration.ArchivingSettings.DEFAULT_SEARCH_PAGE_SIZE
import com.xebialabs.xlrelease.db.ArchivedReleases
import com.xebialabs.xlrelease.utils.ArchivedReleaseExporter
import grizzled.slf4j.Logging
import org.joda.time.LocalDateTime
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.JdbcTemplate
import org.springframework.stereotype.Service

import java.io.File
import java.util.Date
import java.util.concurrent.atomic.{AtomicBoolean, AtomicInteger}
import javax.annotation.PreDestroy
import scala.util.{Failure, Try}

@Service
class ArchivePurgingScheduleService @Autowired()(val archivingService: ArchivingService,
                                                 val archivingConfig: ArchivingSettingsManager,
                                                 @Qualifier("reportingJdbcTemplate")
                                                 val reportingJdbcTemplate: JdbcTemplate,
                                                 val archivedReleases: ArchivedReleases) extends Logging {

  private val isCancelRequested = new AtomicBoolean(false)
  private val purgeInProgressLock = new Object

  def getCronSchedule: String = archivingConfig.getPurgingJobCronSchedule

  @Timed
  def purgeArchive(): Unit =
    purgeArchive(archivingConfig.getMaximumArchiveRetentionPeriod)

  @Timed
  def purgeArchive(retentionPeriod: Integer): Unit = {
    if (archivingConfig.getPurgingEnabled) {
      val startPurging = System.currentTimeMillis()

      val expirationDate = LocalDateTime.now().minusHours(retentionPeriod).toDate
      logger.debug(s"Purging archived releases older than $retentionPeriod hours (ended before $expirationDate)...")

      purgeInProgressLock.synchronized {
        val count = findAndProcessPurgableReleases(expirationDate)
        logger.debug(s"Purged $count releases in ${System.currentTimeMillis() - startPurging} ms")
      }
    } else {
      logger.debug("Purging archive is not enabled.")
    }
  }

  private def createRootDir(path: String): File = {
    val rootDir = new File(path)

    if (!rootDir.exists()) {
      try {
        rootDir.mkdirs()
        logger.debug(s"Created root directory for purged releases at '${rootDir.getAbsolutePath}' based on provided path '$path'")
      } catch {
        case e: Exception =>
          val msg = s"Unable to create root directory for purged releases '$path'."
          throw new IllegalStateException(msg, e)
      }
    }

    if (!rootDir.isDirectory) {
      val msg = s"Purged release storage root directory path '$path' does not point to a directory."
      throw new IllegalStateException(msg)
    }

    rootDir
  }

  private def getRootDir(): Option[File] = {
    if (archivingConfig.getExportOnPurgeEnabled()) {
      Some(createRootDir(archivingConfig.getPurgeExportPath()))
    } else {
      None
    }
  }

  private def findAndProcessPurgableReleases(date: Date): Int = {
    val rootDir = getRootDir()
    val runStartTimeMillis = System.currentTimeMillis()

    val count = new AtomicInteger(0)
    var done = false

    while (!done && !isCancelRequested.get()) {
      val pageSize: Int = getSearchPageSize

      logger.debug("Searching for releases that can be purged from the archive")
      val releaseIds = archivingService.findPurgableReleaseIds(date, pageSize)

      if (releaseIds.nonEmpty && count.get() == 0) {
        logger.info(s"Purging releases that ended before $date")
      }

      done = processReleases(releaseIds, count, runStartTimeMillis, rootDir)

      if (!done) {
        if (releaseIds.isEmpty || releaseIds.size < pageSize) {
          logger.debug(s"Processed only ${releaseIds.size} finished releases on this page: we're done.")
          done = true
        } else {
          logger.debug(s"Purged ${count.get()} releases that completed before $date, fetching the next $pageSize")
        }
      }
    }

    if (count.get() > 0) {
      logger.info(s"Purged ${count.get()} releases that completed before $date")
    }
    count.get()
  }

  //noinspection ScalaStyle
  private def processReleases(releaseIds: Iterable[String], count: AtomicInteger, runStartTimeMillis: Long, rootDir: Option[File]): Boolean = {
    for (releaseId <- releaseIds) {
      if (isCancelRequested.get()) {
        return true
      }

      if (count.get() > 0) {
        sleepIfNeeded()
      }

      purgeRelease(releaseId, rootDir)
      count.incrementAndGet()

      if (ranTooLong(runStartTimeMillis)) {
        logger.warn(s"Purging paused because it ran for more than ${archivingConfig.getMaxSecondsPerRun} seconds. " +
          s"Purging will continue in the next run")
        return true
      }
    }

    false
  }

  private def purgeRelease(releaseId: String, rootDir: Option[File]): Try[Unit] = {
    Try {
      rootDir match {
        case Some(dir) =>
          val exporter = new ArchivedReleaseExporter(reportingJdbcTemplate, dir, archivedReleases)
          val path = exporter.exportReleaseData(releaseId)
          logger.info(s"Purging release [$releaseId] from the archive, data exported to ${path}")
        case None =>
      }
      val s = System.currentTimeMillis()
      archivingService.purgeArchivedRelease(releaseId)
      logger.debug(s"Purged release [$releaseId] from the archive in ${System.currentTimeMillis() - s} ms")
    }.recoverWith {
      case e: Exception =>
        logger.error(s"Could not purge release [$releaseId] from the archive: ${e.getMessage}", e)
        Failure(e)
    }
  }

  private def getSearchPageSize: Int =
    if (archivingConfig.getPurgingSearchPageSize > 0) archivingConfig.getPurgingSearchPageSize else DEFAULT_SEARCH_PAGE_SIZE

  private def sleepIfNeeded(): Unit = {
    if (archivingConfig.getPurgingSleepSecondsBetweenReleases > 0) {
      logger.debug(s"Sleeping for ${archivingConfig.getPurgingSleepSecondsBetweenReleases} seconds before purging the next release")
      try {
        Thread.sleep(archivingConfig.getPurgingSleepSecondsBetweenReleases * 1000)
      } catch {
        case _: InterruptedException =>
          logger.warn(s"Sleeping for ${archivingConfig.getPurgingSleepSecondsBetweenReleases} seconds was interrupted")
      }
    }
  }

  private def ranTooLong(runStartTimeMillis: Long): Boolean = {
    if (archivingConfig.getPurgingMaxSecondsPerRun > 0) {
      (System.currentTimeMillis() - runStartTimeMillis) > (archivingConfig.getPurgingMaxSecondsPerRun * 1000)
    } else {
      false
    }
  }

  private def off(): Unit = {
    isCancelRequested.set(true)
    logger.debug("Waiting until no purging is in progress")
    purgeInProgressLock.synchronized {
      logger.debug("Purging done, destroying ArchivePurgingScheduleService")
    }
  }

  @PreDestroy
  private def preDestroy(): Unit = {
    off()
  }
}
