package com.xebialabs.xlrelease.service

import com.xebialabs.xlrelease.domain.distributed.events._
import com.xebialabs.xlrelease.events.{AsyncSubscribe, EventListener}
import com.xebialabs.xlrelease.repository.{CiSSERepository, Ids, InMemoryCiSSERepository}
import com.xebialabs.xlrelease.scheduler.logs.TaskLogCreated
import grizzled.slf4j.Logging
import org.jboss.resteasy.plugins.providers.sse.OutboundSseEventImpl
import org.springframework.stereotype.Service

import javax.ws.rs.core.MediaType
import javax.ws.rs.sse.SseEventSink
import scala.collection._
import scala.jdk.FutureConverters.CompletionStageOps


object CiSSEService {
  private val LOG_EVENT = "TASK_LOG_EVENT"
  private val COMMENT_EVENT = "TASK_COMMENT_EVENT"
  private val ATTACHMENT_EVENT = "TASK_ATTACHMENT_EVENT"
  private val TASK_STATUS_LINE = "TASK_STATUS_LINE"
  private val TASK_EXECUTION_EVENT = "TASK_EXECUTION_EVENT"
}

@Service
@EventListener
class CiSSEService extends Logging {

  private val sseRepository: CiSSERepository = new InMemoryCiSSERepository()

  private def newEventBuilder() = new OutboundSseEventImpl.BuilderImpl()

  def followCi(id: String, sink: SseEventSink): Unit = {
    sseRepository.put(id, sink)
  }

  @AsyncSubscribe
  def onEvent(event: DistributedXLReleaseEvent): Unit = {
    event match {
      case TaskLogCreated(taskId, executionId, _) =>
        sendEventToSink(taskId, CiSSEService.LOG_EVENT, executionId)
      case e: DistributedCommentEvent =>
        sendEventToSink(e.taskId, CiSSEService.COMMENT_EVENT, "")
      case e: DistributedAttachmentEvent =>
        sendEventToSink(e.containerId, CiSSEService.ATTACHMENT_EVENT, "")
      case e: DistributedTaskStatusLineUpdated =>
        sendEventToSink(e.taskId, CiSSEService.TASK_STATUS_LINE, e.statusLine)
      case e: DistributedTaskStatusEvent =>
        sendEventToSink(Ids.releaseIdFrom(e.containerId), CiSSEService.TASK_EXECUTION_EVENT, e.taskStatus.value())
      case _ =>
      // nothing to do yet
    }
  }

  private def sendEventToSink(id: String, eventName: String, message: AnyRef): Unit = {
    val sinks = sseRepository.get(id)
    sendEvent(sinks, eventName, message)
  }

  private def sendEvent(sinks: Set[SseEventSink], eventName: String, message: AnyRef): Unit = {
    sinks.foreach(sink => {
      if (!sink.isClosed) {
        val event = newEventBuilder().name(eventName).mediaType(MediaType.APPLICATION_JSON_TYPE).data(message.getClass, message).build()
        sink.send(event).asScala.recover { e =>
          logger.debug("can't send SSE event", e)
          sink.close()
        }(scala.concurrent.ExecutionContext.parasitic)
      }
    })
  }
}
