package com.xebialabs.xlrelease.support.tools

import akka.actor.Actor.Receive
import akka.actor.{Actor, ActorRef}
import com.xebialabs.deployit.checks.Checks.checkArgument
import com.xebialabs.xlrelease.actors.ReleaseExecutionActorMessages.ExtensionCommand
import com.xebialabs.xlrelease.actors.extension.ActorExtensionHandlerFactory
import com.xebialabs.xlrelease.actors.utils.ReleaseActorLifecycleUtils
import com.xebialabs.xlrelease.actors.{ReleaseActorService, ReleaseExecutionActor}
import com.xebialabs.xlrelease.db.ArchivedReleases
import com.xebialabs.xlrelease.domain.status.TaskStatus
import com.xebialabs.xlrelease.domain.status.TaskStatus._
import com.xebialabs.xlrelease.domain.{Release, Task}
import com.xebialabs.xlrelease.exception.LogFriendlyNotFoundException
import com.xebialabs.xlrelease.repository.Ids.releaseIdFrom
import com.xebialabs.xlrelease.repository.ReleaseRepository
import com.xebialabs.xlrelease.service.{ArchivingService, ExecutionService, TaskService}
import com.xebialabs.xlrelease.support.tools.SupportUtilitiesActorExtensionHandlerFactory.{COMMENT_TEXT, ForceTaskStatus}
import com.xebialabs.xlrelease.user.User.AUTHENTICATED_USER
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.{Component, Service}

import java.util.concurrent.TimeUnit
import scala.concurrent.duration.FiniteDuration

@Service
class SupportUtilities @Autowired()(val archivingService: ArchivingService,
                                    val releaseRepository: ReleaseRepository,
                                    val archivedReleases: ArchivedReleases,
                                    val releaseActorLifecycleUtils: ReleaseActorLifecycleUtils,
                                    releaseActorService: ReleaseActorService
                                   ) extends Logging {

  def updateTaskStatus(taskId: String, newStatus: TaskStatus): Unit = {
    val releaseId = releaseIdFrom(taskId)
    checkArgument(releaseRepository.exists(releaseId), s"Release [$releaseId] not found")
    releaseActorService.executeCommand(taskId, ForceTaskStatus(taskId, newStatus))
  }

  def getReleaseContent(releaseId: String): String = {
    if (releaseRepository.exists(releaseId)) {
      releaseRepository.getReleaseJson(releaseId)
    } else {
      archivedReleases.getRelease(releaseId)
        .getOrElse(throw new LogFriendlyNotFoundException(s"Release [$releaseId] not found"))
    }
  }

  def deleteRelease(releaseId: String): Unit = {
    checkArgument(releaseRepository.exists(releaseId) || archivedReleases.exists(releaseId), s"Release [$releaseId] not found")
    try releaseActorLifecycleUtils.terminateReleaseActorAndAwait(releaseId, FiniteDuration.apply(5, TimeUnit.SECONDS))
    catch {
      case e: Exception =>
        logger.error(s"Could not terminate release actor $releaseId within timeout", e)
    }
    archivedReleases.deleteReleaseFromArchive(releaseId)
    if (releaseRepository.exists(releaseId)) {
      try archivingService.archiveAllIncomingDependencies(releaseId)
      catch {
        case e: Exception =>
          logger.error(s"Could not archive incoming dependencies to $releaseId", e)
      }
      releaseRepository.delete(releaseId, failIfReferenced = false)
    }
  }

}

object SupportUtilitiesActorExtensionHandlerFactory {

  case class ForceTaskStatus(taskId: String, newStatus: TaskStatus) extends ExtensionCommand

  val COMMENT_TEXT: TaskStatus => String =
    (newStatus: TaskStatus) => s"Forced transition to new status '${newStatus.toString.toLowerCase.capitalize}' performed by support"
}

@Component
class SupportUtilitiesActorExtensionHandlerFactory @Autowired()(executionService: ExecutionService,
                                                                taskService: TaskService)
  extends ActorExtensionHandlerFactory with Logging {


  override def getHandler(self: ActorRef, sender: () => ActorRef, release: => Release): Receive = {
    case ForceTaskStatus(taskId: String, newStatus: TaskStatus) => replyOrFail(sender) {
      val task = taskService.findById[Task](taskId)
      newStatus match {
        case SKIPPED =>
          val release = task.getRelease
          if (task.isPlanned) {
            executionService.markTaskAsDone(release, SKIPPED_IN_ADVANCE, taskId, COMMENT_TEXT(SKIPPED_IN_ADVANCE), AUTHENTICATED_USER)
          } else {
            executionService.markTaskAsDone(release, SKIPPED, taskId, COMMENT_TEXT(SKIPPED), AUTHENTICATED_USER)
          }
        case COMPLETED =>
          val release = task.getRelease
          if (task.isPlanned) {
            executionService.markTaskAsDone(release, COMPLETED_IN_ADVANCE, taskId, COMMENT_TEXT(COMPLETED_IN_ADVANCE), AUTHENTICATED_USER)
          } else {
            executionService.markTaskAsDone(release, COMPLETED, taskId, COMMENT_TEXT(COMPLETED), AUTHENTICATED_USER)
          }
        case ABORTED =>
          executionService.abortTask(task, COMMENT_TEXT(ABORTED))
        case FAILED =>
          executionService.fail(task, COMMENT_TEXT(FAILED), AUTHENTICATED_USER)
        case _ => throw new IllegalArgumentException(s"Forced transition to status $newStatus not supported")
      }
    }
  }

  override def supports(clazz: Class[_ <: Actor]): Boolean = classOf[ReleaseExecutionActor].isAssignableFrom(clazz)
}

