package com.xebialabs.xlrelease

import akka.actor.{ActorRef, ActorRefFactory}
import com.google.common.base.Charsets
import com.xebialabs.xlrelease.domain._
import com.xebialabs.xlrelease.domain.status.TaskStatus
import com.xebialabs.xlrelease.repository.Ids

import java.net.URLEncoder

package object actors {

  type ReleaseActorMaker = (ActorRefFactory, String) => ActorRef

  type ReleaseExecutorActorMaker = (ActorRefFactory, ActorRef, String) => ActorRef

  type RecoveryActorMaker = (ActorRefFactory, ActorRef, ActorRef, ActorRef) => ActorRef

  type TaskPredicate = Task => Boolean

  def releaseId2ActorName(releaseId: String) = URLEncoder.encode(releaseId.releaseActorId, Charsets.UTF_8.name())

  val inProgressTask: TaskPredicate = task => task.isInProgress

  val inProgressAutomatedTask: TaskPredicate = task => task.isAutomated && !task.isInstanceOf[TaskGroup] && task.isInProgress

  val preconditionInProgressTask: TaskPredicate = task => task.getStatus == TaskStatus.PRECONDITION_IN_PROGRESS

  val facetCheckInProgressTask: TaskPredicate = task => task.getStatus == TaskStatus.FACET_CHECK_IN_PROGRESS

  val failureHandlerInProgressTask: TaskPredicate = task => task.getStatus == TaskStatus.FAILURE_HANDLER_IN_PROGRESS

  object TaskPredicate {

    implicit class TaskPredicateOperations(val taskPredicate: TaskPredicate) extends AnyVal {

      def or(taskPredicate2: TaskPredicate): TaskPredicate = task => taskPredicate(task) || taskPredicate2(task)

      def and(taskPredicate2: TaskPredicate): TaskPredicate = task => taskPredicate(task) && taskPredicate2(task)
    }

  }

  implicit class ReleaseIdActorExtension(id: String) {
    def releaseActorId: String = Ids.releaseActorId(id)
  }

}
