package com.xebialabs.xlrelease.api.internal;

import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.StreamingOutput;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;

import com.xebialabs.xlrelease.api.utils.ResponseHelper;
import com.xebialabs.xlrelease.api.v1.ReleaseApi;
import com.xebialabs.xlrelease.calendar.ReleaseCalendar;
import com.xebialabs.xlrelease.domain.Release;
import com.xebialabs.xlrelease.param.IdParam;
import com.xebialabs.xlrelease.security.PermissionChecker;
import com.xebialabs.xlrelease.service.ReleaseService;

/**
 * Provides release exports in various formats.
 */
@Path("/export")
@Controller
public class ExportResource {

    private static final String ICS_CONTENT_TYPE = "text/calendar";
    private static final String ICS_EXTENSION = ".ics";

    private final PermissionChecker permissions;
    private final ReleaseService releaseService;
    private final ReleaseApi releaseApi;

    @Autowired
    public ExportResource(PermissionChecker permissions,
                          ReleaseService releaseService, ReleaseApi releaseApi) {
        this.permissions = permissions;
        this.releaseService = releaseService;
        this.releaseApi = releaseApi;
    }

    @GET
    @Path("attachments/{attachmentId}")
    @Produces(MediaType.APPLICATION_OCTET_STREAM)
    public Response downloadAttachment(@PathParam("attachmentId") @IdParam String attachmentId) {
        return releaseApi.downloadAttachment(attachmentId);
    }

    @GET
    @Path("calendar/{releaseId}")
    @Produces(MediaType.APPLICATION_OCTET_STREAM)
    public Response exportReleaseToCalendar(@PathParam("releaseId") @IdParam String releaseId) {
        Release release = releaseService.findByIdIncludingArchived(releaseId);
        permissions.checkView(release);

        releaseService.decorateRemovingUnnecessaryFields(release);

        final String calendarContent = new ReleaseCalendar(release).toString();

        StreamingOutput output = outputStream -> outputStream.write(calendarContent.getBytes("UTF-8"));

        String fileName = release.getTitle() + ICS_EXTENSION;
        return ResponseHelper.streamFile(fileName, output, ICS_CONTENT_TYPE);
    }

}
