package com.xebialabs.xlrelease.db


import com.xebialabs.xlrelease.db.ArchivedReleases._
import com.xebialabs.xlrelease.domain.status.FlagStatus
import com.xebialabs.xlrelease.domain.{Release, Team}
import com.xebialabs.xlrelease.security.XLReleasePermissions
import com.xebialabs.xlrelease.service.ArchivingService.getMonthYear
import com.xebialabs.xlrelease.service.PhaseService.DEFAULT_TITLE
import org.joda.time.DateTime
import org.springframework.dao.EmptyResultDataAccessException
import org.springframework.jdbc.core.{BatchPreparedStatementSetter, PreparedStatementCallback, RowMapper}

import java.io.ByteArrayInputStream
import java.nio.charset.StandardCharsets
import java.sql.{PreparedStatement, Timestamp, Types}
import java.util.Date
import scala.jdk.CollectionConverters._
import scala.language.implicitConversions


trait ArchivedReleasesHelpers { self: ArchivedReleases =>

  //noinspection ScalaStyle
  protected def insertRelease(release: Release, releaseJson: String, activityLogs: String, preArchived: Boolean): Boolean = {
    jdbcTemplate.execute(
      s"""INSERT INTO $REPORT_RELEASES_TABLE_NAME(
                          $REPORT_RELEASES_ID_COLUMN,
                          $REPORT_RELEASES_RELEASEJSON_COLUMN,
                          $REPORT_RELEASES_LOGSJSON_COLUMN,
                          $REPORT_RELEASES_TITLE_COLUMN,
                          $REPORT_RELEASES_START_DATE_COLUMN,
                          $REPORT_RELEASES_END_DATE_COLUMN,
                          $REPORT_RELEASES_DURATION_COLUMN,
                          $REPORT_RELEASES_MONTH_YEAR_COLUMN,
                          $REPORT_RELEASES_OWNER_COLUMN,
                          $REPORT_RELEASES_STATUS_COLUMN,
                          $REPORT_RELEASES_MANUAL_TASKS_COUNT_COLUMN,
                          $REPORT_RELEASES_AUTOMATED_TASKS_COUNT_COLUMN,
                          $REPORT_RELEASES_MANUAL_TASKS_DURATION_COLUMN,
                          $REPORT_RELEASES_AUTOMATED_TASKS_DURATION_COLUMN,
                          $REPORT_RELEASES_IS_FLAGGED_COLUMN,
                          $REPORT_RELEASES_ORIGIN_TEMPLATE_ID,
                          $REPORT_RELEASES_PRE_ARCHIVED,
                          $REPORT_RELEASES_KIND)
                          values(?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?)""",
      new PreparedStatementCallback[Boolean] {
        override def doInPreparedStatement(ps: PreparedStatement): Boolean = {
          try {
            val monthYear = Option(release.getEndDate).map(d => getMonthYear(new DateTime(d))).orNull

            val automatedTasks = release.getAllTasks.asScala.filter(_.isAutomated)
            val manualTasks = release.getAllTasks.asScala.filter(!_.isAutomated)
            val automatedTasksDurations = automatedTasks.flatMap(task => durationOption(task.getStartDate, task.getEndDate))
            val manualTasksDurations = manualTasks.flatMap(task => durationOption(task.getStartDate, task.getEndDate))

            ps.setString(1, shortenId(release.getId))
            ps.setBinaryStream(2, new ByteArrayInputStream(releaseJson.getBytes(StandardCharsets.UTF_8)))
            ps.setBinaryStream(3, new ByteArrayInputStream(activityLogs.getBytes(StandardCharsets.UTF_8)))
            ps.setString(4, truncate(release.getTitle, COLUMN_LENGTH_TITLE))
            ps.setTimestamp(5, release.getStartDate)
            ps.setTimestamp(6, release.getEndDate)
            durationOption(release.getStartDate, release.getEndDate) match {
              case Some(duration) => ps.setLong(7, duration)
              case None => ps.setNull(7, Types.BIGINT)
            }
            ps.setString(8, monthYear)
            ps.setString(9, truncate(release.getOwner, COLUMN_LENGTH_AUTHORITY_NAME))
            ps.setString(10, release.getStatus.value())
            ps.setInt(11, manualTasks.size)
            ps.setInt(12, automatedTasks.size)
            ps.setLong(13, manualTasksDurations.sum)
            ps.setLong(14, automatedTasksDurations.sum)
            // intentionally not using release.isFlagged() here as its meaning is different
            ps.setInt(15, if (release.getRealFlagStatus == FlagStatus.OK) 0 else 1)
            ps.setString(16, originalTemplateIdToColumnValue(release.getOriginTemplateId))
            ps.setInt(17, preArchived.asInteger)
            ps.setString(18, release.getKind.value())
            ps.execute()
          } catch {
            case e: Throwable =>
              throw new ReleaseArchivalException(s"Failed to archive release ${release.getId}", e)
          }
        }
      })
  }

  protected def insertPhases(release: Release): Unit = {

    val columns = Seq(REPORT_PHASES_ID_COLUMN,
      REPORT_PHASES_TITLE_COLUMN,
      REPORT_PHASES_DURATION_COLUMN,
      REPORT_PHASES_START_DATE_COLUMN,
      REPORT_PHASES_END_DATE_COLUMN,
      REPORT_PHASES_STATUS_COLUMN,
      REPORT_PHASES_RELEASEID_COLUMN,
      REPORT_PHASES_RELEASE_TITLE_COLUMN,
      REPORT_PHASES_RELEASE_OWNER_COLUMN
    )

    val query = s"INSERT INTO $REPORT_PHASES_TABLE_NAME (${columns.mkString(", ")}) VALUES(${columns.map(_ => "?").mkString(", ")})"
    jdbcTemplate.batchUpdate(query, new BatchPreparedStatementSetter {

      override def setValues(ps: PreparedStatement, i: Int): Unit = {
        val phase = release.getPhase(i)

        ps.setString(1, shortenId(phase.getId))
        ps.setString(2, truncate(Option(phase.getTitle).filter(_.nonEmpty).getOrElse(DEFAULT_TITLE), COLUMN_LENGTH_TITLE))
        durationOption(phase.getStartDate, phase.getEndDate) match {
          case Some(duration) => ps.setLong(3, duration)
          case None => ps.setNull(3, Types.BIGINT)
        }
        ps.setTimestamp(4, phase.getStartDate)
        ps.setTimestamp(5, phase.getEndDate)
        ps.setString(6, phase.getStatus.value())
        ps.setString(7, shortenId(release.getId))
        ps.setString(8, truncate(release.getTitle, COLUMN_LENGTH_TITLE))
        ps.setString(9, truncate(release.getOwner, COLUMN_LENGTH_AUTHORITY_NAME))
      }

      override def getBatchSize: Int = release.getPhases.size()
    })
  }

  protected def insertTasks(release: Release): Unit = {

    val columns = Seq(
      REPORT_TASKS_ID_COLUMN,
      REPORT_TASKS_TITLE_COLUMN,
      REPORT_TASKS_DURATION_COLUMN,
      REPORT_TASKS_START_DATE_COLUMN,
      REPORT_TASKS_END_DATE_COLUMN,
      REPORT_TASKS_STATUS_COLUMN,
      REPORT_TASKS_TEAM_COLUMN,
      REPORT_TASKS_OWNER_COLUMN,
      REPORT_TASKS_PHASE_TITLE_COLUMN,
      REPORT_TASKS_RELEASEID_COLUMN,
      REPORT_TASKS_RELEASE_TITLE_COLUMN,
      REPORT_TASKS_RELEASE_OWNER_COLUMN,
      REPORT_TASKS_RELEASE_STATUS_COLUMN,
      REPORT_TASKS_IS_AUTOMATED_COLUMN,
      REPORT_TASKS_TASK_TYPE_COLUMN
    )

    val query = s"INSERT INTO $REPORT_TASKS_TABLE_NAME (${columns.mkString(", ")}) VALUES(${columns.map(_ => "?").mkString(", ")})"
    jdbcTemplate.batchUpdate(query, new BatchPreparedStatementSetter {

      override def setValues(ps: PreparedStatement, i: Int): Unit = {
        val task = release.getAllTasks.get(i)

        ps.setString(1, shortenId(task.getId))
        ps.setString(2, truncate(task.getTitle, COLUMN_LENGTH_TITLE))
        durationOption(task.getStartDate, task.getEndDate) match {
          case Some(duration) => ps.setLong(3, duration)
          case None => ps.setNull(3, Types.BIGINT)
        }
        ps.setTimestamp(4, task.getStartDate)
        ps.setTimestamp(5, task.getEndDate)
        ps.setString(6, task.getStatus.value())
        ps.setString(7, truncate(task.getTeam, COLUMN_LENGTH_AUTHORITY_NAME))
        ps.setString(8, truncate(task.getOwner, COLUMN_LENGTH_AUTHORITY_NAME))
        ps.setString(9, truncate(task.getPhase.getTitle, COLUMN_LENGTH_TITLE))
        ps.setString(10, shortenId(release.getId))
        ps.setString(11, truncate(release.getTitle, COLUMN_LENGTH_TITLE))
        ps.setString(12, truncate(release.getOwner, COLUMN_LENGTH_AUTHORITY_NAME))
        ps.setString(13, release.getStatus.value())
        ps.setInt(14, task.isAutomated.asInteger)
        ps.setString(15, task.getTaskType.toString)
      }

      override def getBatchSize: Int = release.getAllTasks.size()
    })

  }

  protected def insertTags(release: Release): Array[Int] = {
    val distinctTags = release.getTags.asScala.map(_.toLowerCase).distinct
    jdbcTemplate.batchUpdate(s"INSERT INTO $ARCHIVE_TAGS_TABLE_NAME($TAGS_RELEASEID_COLUMN, $TAGS_VALUE_COLUMN) VALUES(?, ?)",
      new BatchPreparedStatementSetter {
        override def setValues(ps: PreparedStatement, i: Int): Unit = {
          ps.setString(1, shortenId(release.getId))
          ps.setString(2, truncate(distinctTags(i), COLUMN_LENGTH_TITLE))
        }

        override def getBatchSize: Int = distinctTags.size
      })
  }

  private val STMT_INSERT_MEMBER: String =
    s"""|INSERT INTO $ARCHIVE_MEMBERVIEWERS_TABLE_NAME (
        |   $ARCHIVE_MEMBERVIEWERS_RELEASEID_COLUMN,
        |   $ARCHIVE_MEMBERVIEWERS_USERNAME_COLUMN
        | )
        | VALUES (?, ?)""".stripMargin

  protected def insertMemberViewers(releaseId: String, effectiveMembers: Seq[String]): Array[Int] = {
    jdbcTemplate.batchUpdate(STMT_INSERT_MEMBER,
      new BatchPreparedStatementSetter {
        override def setValues(ps: PreparedStatement, i: Int): Unit = {
          ps.setString(1, shortenId(releaseId))
          ps.setString(2, truncate(effectiveMembers(i), COLUMN_LENGTH_AUTHORITY_NAME))
        }

        override def getBatchSize: Int = effectiveMembers.size
      })
  }

  private val STMT_INSERT_ROLE: String =
    s"""|INSERT INTO $ARCHIVE_ROLEVIEWERS_TABLE_NAME (
        |   $ARCHIVE_ROLEVIEWERS_RELEASEID_COLUMN,
        |   $ARCHIVE_ROLEVIEWERS_ROLE_COLUMN
        | )
        | VALUES (?, ?)""".stripMargin

  protected def insertRoleViewers(releaseId: String, effectiveRoles: Seq[String]): Array[Int] = {
    jdbcTemplate.batchUpdate(STMT_INSERT_ROLE,
      new BatchPreparedStatementSetter {
        override def setValues(ps: PreparedStatement, i: Int): Unit = {
          ps.setString(1, shortenId(releaseId))
          ps.setString(2, truncate(effectiveRoles(i), COLUMN_LENGTH_AUTHORITY_NAME))
        }

        override def getBatchSize: Int = effectiveRoles.size
      })
  }

  private val STMT_DELETE_MEMBER: String =
    s"""|DELETE FROM $ARCHIVE_MEMBERVIEWERS_TABLE_NAME
        | WHERE $ARCHIVE_MEMBERVIEWERS_RELEASEID_COLUMN = ?
        |   AND $ARCHIVE_MEMBERVIEWERS_USERNAME_COLUMN = ?""".stripMargin

  protected def deleteMemberViewers(releaseId: String, effectiveMembers: Seq[String]): Array[Int] = {
    jdbcTemplate.batchUpdate(STMT_DELETE_MEMBER,
      new BatchPreparedStatementSetter {
        override def setValues(ps: PreparedStatement, i: Int): Unit = {
          ps.setString(1, shortenId(releaseId))
          ps.setString(2, truncate(effectiveMembers(i), COLUMN_LENGTH_AUTHORITY_NAME))
        }

        override def getBatchSize: Int = effectiveMembers.size
      })
  }

  private val STMT_DELETE_ROLE: String =
    s"""|DELETE FROM $ARCHIVE_ROLEVIEWERS_TABLE_NAME
        | WHERE $ARCHIVE_ROLEVIEWERS_RELEASEID_COLUMN = ?
        |   AND $ARCHIVE_ROLEVIEWERS_ROLE_COLUMN = ?""".stripMargin

  protected def deleteRoleViewers(releaseId: String, effectiveRoles: Seq[String]): Array[Int] = {
    jdbcTemplate.batchUpdate(STMT_DELETE_ROLE,
      new BatchPreparedStatementSetter {
        override def setValues(ps: PreparedStatement, i: Int): Unit = {
          ps.setString(1, shortenId(releaseId))
          ps.setString(2, truncate(effectiveRoles(i), COLUMN_LENGTH_AUTHORITY_NAME))
        }

        override def getBatchSize: Int = effectiveRoles.size
      })
  }

  protected def getReleaseBlobColumnAsString(releaseId: String, columnName: String, includePreArchived: Boolean): Option[String] = {
    queryReleaseField(releaseId, columnName, includePreArchived, releaseContentMapper)
  }

  protected def queryReleaseField[T](releaseId: String, columnName: String, includePreArchived: Boolean, rowMapper: RowMapper[T]): Option[T] = {
    val query = s"SELECT $columnName FROM $REPORT_RELEASES_TABLE_NAME WHERE $REPORT_RELEASES_ID_COLUMN = ?" + {
      if (!includePreArchived) {
        s" AND $REPORT_RELEASES_PRE_ARCHIVED = 0"
      } else {
        ""
      }
    }
    try {
      Some(jdbcTemplate.queryForObject(query, Array[Object](shortenId(releaseId)), rowMapper))
    } catch {
      case _: EmptyResultDataAccessException => None
    }
  }

  protected def truncate(value: String, size: Int): String = Option(value).map(_.take(size)).orNull

  protected def collectTeamsPropertiesWithViewPermissions(teamList: Seq[Team], properties: Team => Seq[String]): Seq[String] = {
    teamList
      .filter(team => team.getPermissions.contains(XLReleasePermissions.VIEW_RELEASE.getPermissionName))
      .flatMap(properties).distinct
  }

  private implicit def dateToNullableTimestamp(date: Date): Timestamp =
    Option(date).map(d => new Timestamp(d.getTime)).orNull

  private def durationOption(startDate: Date, endDate: Date): Option[Long] = {
    (Option(startDate), Option(endDate)) match {
      case (Some(start), Some(end)) =>
        Some(new DateTime(end).getMillis - new DateTime(start).getMillis)
      case _ =>
        None
    }
  }

  protected def deleteRelease(releaseId: String): Boolean = {
    jdbcTemplate.update(
      s"""|DELETE FROM $REPORT_RELEASES_TABLE_NAME
          | WHERE $REPORT_RELEASES_ID_COLUMN = ?""".stripMargin,
      (ps: PreparedStatement) => {
        ps.setString(1, shortenId(releaseId))
      }
    ) == 1
  }

}
