package com.xebialabs.xlrelease.domain.tasks.customscript;

import java.util.Set;
import org.springframework.stereotype.Component;

import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.xlrelease.domain.*;
import com.xebialabs.xlrelease.domain.tasks.TaskUpdateDirective;
import com.xebialabs.xlrelease.domain.tasks.task.DefaultTaskUpdater;
import com.xebialabs.xlrelease.utils.PasswordVerificationUtils;

import static com.google.common.base.Preconditions.checkArgument;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.CI;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.STRING;
import static com.xebialabs.xlrelease.domain.Task.CATEGORY_OUTPUT;
import static com.xebialabs.xlrelease.domain.tasks.TaskUpdateDirective.UPDATE_TASK_CONFIGURATION;
import static com.xebialabs.xlrelease.variable.VariableHelper.formatVariableIfNeeded;

@Component
public class CustomScriptTaskUpdater extends DefaultTaskUpdater {

    @Override
    public Class<? extends Task> getTaskClass() {
        return CustomScriptTask.class;
    }

    @Override
    public Changes update(final Task original, final Task updated, final Set<TaskUpdateDirective> updateDirectives) {
        Changes changes = super.update(original, updated, updateDirectives);
        if (hasDirectiveToUpdateAllProperties(original, updateDirectives)) {

            CustomScriptTask updatedCustomScriptTask = ((CustomScriptTask) updated);
            PythonScript updatedPythonScript = updatedCustomScriptTask.getPythonScript();
            CustomScriptTask originalCustomScriptTask = (CustomScriptTask) original;
            PythonScript originalPythonScript = originalCustomScriptTask.getPythonScript();

            updateCustomScriptTaskConfiguration(originalPythonScript, updatedPythonScript);
            originalCustomScriptTask.setKeepPreviousOutputPropertiesOnRetry(updatedCustomScriptTask.isKeepPreviousOutputPropertiesOnRetry());
            changes.update(originalPythonScript);
        } else if (updateDirectives.contains(UPDATE_TASK_CONFIGURATION)) {
            changes.update(original);

            CustomScriptTask updatedCustomScriptTask = ((CustomScriptTask) updated);
            PythonScript updatedPythonScript = updatedCustomScriptTask.getPythonScript();
            CustomScriptTask originalCustomScriptTask = (CustomScriptTask) original;
            PythonScript originalPythonScript = originalCustomScriptTask.getPythonScript();

            updateCustomScriptTaskConfiguration(originalPythonScript, updatedPythonScript);
        }

        return changes;
    }

    private void updateCustomScriptTaskConfiguration(final PythonScript original, final PythonScript updatedPythonScript) {
        updateInputProperties(original, updatedPythonScript);
        updateOutputProperties(original, updatedPythonScript);
    }

    private void updateInputProperties(final PythonScript original, final PythonScript updated) {
        for (PropertyDescriptor propertyDescriptor : original.getInputProperties()) {
            String name = propertyDescriptor.getName();
            if (propertyDescriptor.getKind() == CI) {
                Configuration configuration = updated.getProperty(name);
                if (configuration != null) {
                    checkArgument(
                            configuration.getType().getDescriptor().isAssignableTo(propertyDescriptor.getReferencedType()),
                            "Type of CI must be " + propertyDescriptor.getReferencedType().getName() + " got " + configuration.getType().getName()
                    );
                }
                original.setProperty(propertyDescriptor.getName(), configuration);
            } else if (propertyDescriptor.isPassword() && propertyDescriptor.getKind() == STRING) {
                original.setProperty(propertyDescriptor.getName(), PasswordVerificationUtils.replacePasswordIfNeeded(original.getProperty(name), updated.getProperty(name)));
            } else {
                original.setProperty(propertyDescriptor.getName(), updated.getProperty(name));
            }
        }
    }

    private void updateOutputProperties(final PythonScript original, final PythonScript updated) {
        for (PropertyDescriptor propertyDescriptor : original.getTransitionalAndOutputProperties()) {
            String name = propertyDescriptor.getName();
            Object value = updated.getProperty(name);

            if (propertyDescriptor.getKind() == STRING && CATEGORY_OUTPUT.equals(propertyDescriptor.getCategory())) {
                // backwards compatible, all STRING output properties are wrapped into variable syntax ${..}
                value = formatVariableIfNeeded((String) value);
            }

            original.setProperty(name, value);
        }
    }

}
