package com.xebialabs.xlrelease.scheduler.workers

import com.xebialabs.xlrelease.actors.ReleaseActorService
import com.xebialabs.xlrelease.scheduler.ScriptTaskJob
import com.xebialabs.xlrelease.scheduler.workers.Worker.ExecuteJob
import com.xebialabs.xlrelease.script.{FailureScriptTaskResult, RestartScriptTaskResult, ScriptService, SuccessScriptTaskResult}
import com.xebialabs.xlrelease.service.{CommentService, PhaseService}
import com.xebialabs.xlrelease.user.User.{AUTHENTICATED_USER, SYSTEM}
import org.springframework.stereotype.Component

import scala.util.Try

@Component
class ScriptTaskWorker(val releaseActorService: ReleaseActorService,
                       phaseService: PhaseService,
                       val scriptServicesList: java.util.List[ScriptService],
                       commentService: CommentService
                      )
  extends Worker with ScriptServiceSupport with TaskWorkerFailureLogic {

  override def execute: ExecuteJob = {
    case ScriptTaskJob(taskRef) => doRun(taskRef) {
      val task = taskRef.get()
      commentService.decorate(task)
      val result = scriptServiceFor(task).executeScriptTask(task)
      result match {
        case SuccessScriptTaskResult(taskId, executionId, executionLog, attachmentId, scriptTaskResults, authentication) =>
          withAuthentication(authentication) {
            releaseActorService.finishScriptTask(taskId, executionId, executionLog, attachmentId, Some(scriptTaskResults))
          }
        case RestartScriptTaskResult(taskId, executionId, executionLog, attachmentId, scriptTaskResults, exception, authentication) =>
          withAuthentication(authentication) {
            commentService.create(taskId, exception.getMessage, SYSTEM, false)
            releaseActorService.failScriptTask(taskId, exception.getMessage, executionId, Option.empty, Some(scriptTaskResults))
            Try(phaseService.restartPhase(exception.getReleaseId, exception.getPhaseId, exception.getTaskId, exception.getPhaseVersion, exception.isResumeRelease))
              .recover {
                case e: Exception =>
                  commentService.create(taskId, e.getMessage, AUTHENTICATED_USER, true);
              }
          }
        case FailureScriptTaskResult(taskId, executionId, executionLog, attachmentId, scriptTaskResults, authentication) =>
          withAuthentication(authentication) {
            releaseActorService.failScriptTask(taskId, executionLog, executionId, attachmentId, Option(scriptTaskResults))
          }
      }
    }
  }

}
