package com.xebialabs.xlrelease.service

import com.google.common.base.Preconditions
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.xlrelease.domain._
import com.xebialabs.xlrelease.security.PermissionChecker

import java.util.Objects

object LockedTaskOperationChecks {

  def checkAddWatcher(task: Task): Unit = {
    checkState(!task.isLocked, "You can not add a watcher on a locked task: %s.", task.getTitle)
  }

  def checkRemoveWatcher(task: Task): Unit = {
    checkState(!task.isLocked, "You can not remove a watcher on a locked task: %s.", task.getTitle)
  }

  def checkUpdateWatchers(task: Task): Unit = {
    checkState(!task.isLocked, "You can not update watchers on a locked task: %s.", task.getTitle)
  }

  def checkDeleteAttachment(task: Task): Unit = {
    checkState(!task.isLocked, "You can not delete an attachment from a locked task: %s.", task.getTitle)
  }

  def checkCreateGateCondition(task: Task): Unit = {
    checkState(!task.isLocked, "You can not add a condition to a locked task: %s.", task.getTitle)
  }

  def checkUpdateGateCondition(task: Task, originalGateCondition: GateCondition, updatedGateCondition: GateCondition): Unit = {
    checkState(!task.isLocked || Objects.equals(updatedGateCondition.getTitle, originalGateCondition.getTitle),
      "You can not edit condition on a gate task. You can only check or uncheck a condition.")
  }

  def checkDeleteGateCondition(task: Task): Unit = {
    checkState(!task.isLocked, "You can not delete a condition on a locked task: %s.", task.getTitle)
  }

  def checkCreateDependency(task: Task): Unit = {
    checkState(!task.isLocked, "You can not add dependency to the locked task: %s.", task.getTitle)
  }

  def checkUpdateDependency(task: Task): Unit = {
    checkState(!task.isLocked, "You can not edit dependency on a locked task: %s.", task.getTitle)
  }

  def checkDeleteDependency(task: Task): Unit = {
    checkState(!task.isLocked, "You can not delete dependency from a locked task: %s.", task.getTitle)
  }

  def checkTaskUpdate(original: Task, updatedTask: Task): Unit = {
    checkState(!original.isLocked || !TaskUpdatePropertiesComparator.hasChanged(original, updatedTask),
      "You can not update a locked task: %s.", original.getTitle)
  }

  def checkTaskTypeChange(task: Task): Unit = {
    checkState(!task.isLocked, "You can not change type of the locked task %s.", task.getTitle)
    checkState(!task.getContainer.isLocked,
      "You can not change type of a task %s inside of a locked container %s.",
      task.getTitle, task.getContainer.getTitle)
  }

  def checkUpdateTeams(task: Task): Unit = {
    checkState(!task.isLocked, "You can not update teams of the locked task %s.", task.getTitle)
  }

  def checkUpdateOwner(task: Task, newOwner: String, permissionChecker: PermissionChecker): Unit = {
    checkState(permissionChecker.isCurrentUserAdmin
      || !task.isLocked
      || permissionChecker.areUsersInTheSameTaskTeam(task, newOwner), "You can not update owner of the locked task: %s.", task.getTitle)
  }

  def checkDeleteTask(task: Task): Unit = {
    checkState(!task.isLocked, "You can not delete locked task: %s.", task.getTitle)
    checkState(!task.getContainer.isLocked, "You can not delete a task %s inside of a locked container %s.", task.getTitle, task.getContainer.getTitle)
  }

  def checkMoveTask(task: Task, originTaskContainer: TaskContainer, targetTaskContainer: TaskContainer): Unit = {
    checkState(!originTaskContainer.isLocked,
      "You can not move task %s out of or within the locked container %s.",
      task.getTitle, originTaskContainer.getTitle)
    checkState(!targetTaskContainer.isLocked,
      "You can not move task %s into the locked container %s.",
      task.getTitle, originTaskContainer.getTitle)
    checkState(!task.isLocked || originTaskContainer.equals(targetTaskContainer),
      "Locked task %s cannot be moved between task container %s and %s.",
      task.getTitle, originTaskContainer.getId, targetTaskContainer.getId)
  }

  def checkSkipTask(task: Task): Unit = {
    checkState(!task.isLocked, "You can not skip locked task: %s.", task.getTitle)
  }

  def checkCompleteTaskInAdvance(task: Task): Unit = {
    checkState(!task.isLocked, "You can not complete in advance locked task %s.", task.getTitle)
  }

  def checkCopyTask(task: Task): Unit = {
    checkState(!task.isLocked, "You can not duplicate locked task.")
    checkState(!task.getContainer.isLocked, "You can not duplicate a task %s inside of a locked container %s.", task.getTitle, task.getContainer.getTitle)
  }

  def checkCreateTaskInContainer(container: TaskContainer): Unit = {
    checkState(!container.isLocked, "You can't create a task inside of a locked container %s.", container.getTitle)
  }

  def checkCreateTaskLink(parallelGroup: ParallelGroup): Unit = {
    checkState(!parallelGroup.isLocked, "You can't add a link inside a locked task group %s.", parallelGroup.getTitle)
  }

  def checkRemoveTaskLink(parallelGroup: ParallelGroup): Unit = {
    checkState(!parallelGroup.isLocked, "You can't remove a link from a locked task group %s.", parallelGroup.getTitle)
  }

  private def checkState(expr: Boolean, msg: String, args: Object*): Unit = {
    if (args.isEmpty) {
      Preconditions.checkState(expr, msg.asInstanceOf[Object])
    } else {
      Preconditions.checkState(expr, msg, args: _*)
    }
  }

  object TaskUpdatePropertiesComparator {

    import scala.jdk.CollectionConverters._

    private val allowedProperties = Set("flagStatus", "flagComment")
    private val ignoredProperties = allowedProperties ++ Set("hasBeenFlagged", "pythonScript", "variables")

    def hasChanged(original: Task, updated: Task): Boolean = {
      val havePropertiesChanged = (properties: Iterable[PropertyDescriptor], ci1: ConfigurationItem, ci2: ConfigurationItem) => {
        properties.view
          .filterNot(pd => ignoredProperties(pd.getName) || pd.isAsContainment || pd.isHidden)
          .exists(!_.areEqual(ci1, ci2))
      }

      val properties = original.getType.getDescriptor.getPropertyDescriptors.asScala

      (original, updated) match {
        case (o: CustomScriptTask, u: CustomScriptTask) =>
          val pythonScriptProperties = o.getTaskType.getDescriptor.getPropertyDescriptors.asScala
          havePropertiesChanged(properties, o, u) || havePropertiesChanged(pythonScriptProperties, o.getPythonScript, u.getPythonScript)
        case (o: UserInputTask, u: UserInputTask) =>
          havePropertiesChanged(properties, original, updated) || !o.getVariables.asScala.corresponds(u.getVariables.asScala)(_.getKey == _.getKey)
        case _ => havePropertiesChanged(properties, original, updated)
      }
    }
  }

}
