package com.xebialabs.xlrelease.api.internal;

import javax.ws.rs.*;
import javax.ws.rs.core.MediaType;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import com.codahale.metrics.annotation.Timed;

import com.xebialabs.xlrelease.actors.ReleaseActorService;
import com.xebialabs.xlrelease.domain.Phase;
import com.xebialabs.xlrelease.param.IdParam;
import com.xebialabs.xlrelease.security.PermissionChecker;
import com.xebialabs.xlrelease.service.TaskAccessService;
import com.xebialabs.xlrelease.views.PhaseForm;
import com.xebialabs.xlrelease.views.PhaseFullView;
import com.xebialabs.xlrelease.views.converters.PhaseViewConverter;

import static com.xebialabs.xlrelease.repository.Ids.releaseIdFrom;

/**
 * A phase in a release.
 */
@Path("/phases")
@Consumes({MediaType.APPLICATION_JSON})
@Produces({MediaType.APPLICATION_JSON})
@Controller
public class PhaseResource {
    private ReleaseActorService releaseActorService;

    private TaskAccessService taskAccessService;

    private PermissionChecker permissionChecker;

    private PhaseViewConverter phaseViewConverter;

    @Autowired
    public PhaseResource(ReleaseActorService releaseActorService,
                         TaskAccessService taskAccessService,
                         PermissionChecker permissionChecker,
                         PhaseViewConverter phaseViewConverter) {
        this.releaseActorService = releaseActorService;
        this.taskAccessService = taskAccessService;
        this.permissionChecker = permissionChecker;
        this.phaseViewConverter = phaseViewConverter;
    }

    /**
     * Updates a phase.
     *
     * @param phaseId   the identifier of the phase
     * @param phaseForm the changes to apply
     * @return the phase
     */
    @PUT
    @Timed
    @Path("{phaseId:.*Phase[^/-]*}")
    public PhaseFullView update(@PathParam("phaseId") @IdParam String phaseId, PhaseForm phaseForm) {
        permissionChecker.checkEdit(releaseIdFrom(phaseId));

        Phase phase = releaseActorService.updatePhase(phaseId, phaseForm.toPhase());
        return phaseViewConverter.toFullView(phase, taskAccessService.getAllowedTaskTypesForAuthenticatedUser());
    }

    /**
     * Deletes a phase.
     *
     * @param phaseId the identifier of the phase
     */
    @DELETE
    @Timed
    @Path("{phaseId:.*Phase[^/-]*}")
    public void deletePhase(@PathParam("phaseId") @IdParam String phaseId) {
        permissionChecker.checkEdit(releaseIdFrom(phaseId));
        releaseActorService.deletePhase(phaseId);
    }
}
