package com.xebialabs.xlrelease.api.internal;

import java.util.Comparator;
import java.util.List;
import javax.ws.rs.*;
import javax.ws.rs.core.MediaType;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import com.codahale.metrics.annotation.Timed;

import com.xebialabs.xlrelease.actors.ReleaseActorService;
import com.xebialabs.xlrelease.domain.Team;
import com.xebialabs.xlrelease.param.IdParam;
import com.xebialabs.xlrelease.security.PermissionChecker;
import com.xebialabs.xlrelease.service.TeamService;
import com.xebialabs.xlrelease.views.TeamView;
import com.xebialabs.xlrelease.views.converters.TeamMemberViewConverter;

import static com.xebialabs.deployit.checks.Checks.checkArgument;
import static com.xebialabs.xlrelease.repository.Ids.getParentId;
import static com.xebialabs.xlrelease.repository.Ids.isReleaseId;
import static java.util.Collections.emptyList;
import static java.util.Collections.singletonList;
import static java.util.stream.Collectors.toList;

/**
 * The teams defined on a release.
 */
@Path("/teams")
@Consumes({MediaType.APPLICATION_JSON})
@Produces({MediaType.APPLICATION_JSON})
@Controller
public class TeamResource {

    private final PermissionChecker permissions;
    private final ReleaseActorService releaseActorService;
    private final TeamService teamService;
    private final TeamMemberViewConverter teamMemberViewConverter;

    @Autowired
    public TeamResource(PermissionChecker permissions,
                        ReleaseActorService releaseActorService,
                        TeamService teamService,
                        TeamMemberViewConverter teamMemberViewConverter) {
        this.permissions = permissions;
        this.releaseActorService = releaseActorService;
        this.teamService = teamService;
        this.teamMemberViewConverter = teamMemberViewConverter;
    }

    @GET
    @Timed
    @Path("{teamContainerId:.*(Release|Folder)[^/-]*}")
    public List<TeamView> getTeams(@PathParam("teamContainerId") @IdParam String teamContainerId) {
        permissions.checkViewTeams(teamContainerId);

        return teamService.getEffectiveTeams(teamContainerId).stream()
                .map(team -> new TeamView(team, teamMemberViewConverter))
                .sorted(Comparator.comparing(TeamView::getTeamName))
                .collect(toList());
    }

    @POST
    @Timed
    @Path("{teamContainerId:.*(Release|Folder)[^/-]*}/new")
    public TeamView addTeam(@PathParam("teamContainerId") @IdParam String teamContainerId, TeamView teamView) {
        permissions.checkEditTeams(teamContainerId, singletonList(teamView));
        checkArgument(isReleaseAdminTeamPresent(teamContainerId, singletonList(teamView)),
                "Cannot add team %s in %s, missing 'Release Admin' team", teamView, teamContainerId);

        Team team = teamView.toTeam();
        if (isReleaseId(teamContainerId)) {
            team = releaseActorService.addTeam(teamContainerId, team);
        } else {
            team = teamService.addTeam(teamContainerId, team);
        }

        return new TeamView(team, teamMemberViewConverter);
    }

    @DELETE
    @Timed
    @Path("/{teamId:.*Team[^/-]*}")
    public void deleteTeam(@PathParam("teamId") @IdParam String teamId) {
        String teamContainerId = getParentId(teamId);
        permissions.checkDeleteTeams(teamContainerId, singletonList(teamId));
        if (isReleaseId(teamContainerId)) {
            releaseActorService.deleteTeam(teamId);
        } else {
            teamService.deleteTeam(teamContainerId, teamId);
        }
    }

    @DELETE
    @Timed
    @Path("/{teamContainerId:.*(Release|Folder)[^/-]*}")
    public void deleteAllOwnTeams(@PathParam("teamContainerId") @IdParam String teamContainerId) {
        permissions.checkDeleteOwnTeams(teamContainerId);
        if (isReleaseId(teamContainerId)) {
            releaseActorService.updateTeams(teamContainerId, emptyList());
        } else {
            teamService.deleteTeamsFromPlatform(teamContainerId);
        }
    }

    @PUT
    @Timed
    @Path("{teamId:.*Team[^/-]*}")
    public TeamView updateTeam(@PathParam("teamId") @IdParam String teamId, TeamView newTeam) {
        String teamContainerId = getParentId(teamId);
        permissions.checkEditTeams(teamContainerId, singletonList(newTeam));

        Team team;
        if (isReleaseId(teamContainerId)) {
            team = releaseActorService.updateTeam(teamId, newTeam.toTeam());
        } else {
            team = teamService.updateTeam(teamId, newTeam.toTeam());
        }

        return new TeamView(team, teamMemberViewConverter);
    }

    @POST
    @Timed
    @Path("{teamContainerId:.*(Release|Folder)[^/-]*}")
    public List<TeamView> updateTeams(@PathParam("teamContainerId") @IdParam String teamContainerId, List<TeamView> updatedListOfTeams) {
        permissions.checkEditTeams(teamContainerId, updatedListOfTeams);
        checkArgument(isReleaseAdminTeamPresent(teamContainerId, updatedListOfTeams),
                "Cannot save teams %s in %s, missing 'Release Admin' team", updatedListOfTeams, teamContainerId);

        List<Team> submittedTeams = updatedListOfTeams.stream().map(TeamView::toTeam).collect(toList());
        List<Team> updatedTeams;
        if (isReleaseId(teamContainerId)) {
            updatedTeams = releaseActorService.updateTeams(teamContainerId, submittedTeams);
        } else {
            updatedTeams = teamService.saveTeamsToPlatform(teamContainerId, submittedTeams);
        }
        return updatedTeams.stream().map(team -> new TeamView(team, teamMemberViewConverter)).collect(toList());
    }

    private boolean isReleaseAdminTeamPresent(String containerId, List<TeamView> updatedListOfTeams) {
        return isReleaseAdminTeamPresent(getTeams(containerId)) || isReleaseAdminTeamPresent(updatedListOfTeams);
    }

    private boolean isReleaseAdminTeamPresent(List<TeamView> teamViewList) {
        return teamViewList.stream().anyMatch(team -> Team.RELEASE_ADMIN_TEAMNAME.equals(team.getTeamName()));
    }
}
