package com.xebialabs.xlrelease.api.v1;

import java.util.*;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import com.xebialabs.deployit.security.permission.Permission;
import com.xebialabs.xlrelease.actors.ReleaseActorService;
import com.xebialabs.xlrelease.api.v1.views.TeamView;
import com.xebialabs.xlrelease.domain.Team;
import com.xebialabs.xlrelease.security.PermissionChecker;
import com.xebialabs.xlrelease.service.TeamService;
import com.xebialabs.xlrelease.views.converters.TeamMemberViewConverter;
import com.xebialabs.xlrelease.views.converters.TeamViewConverter;

import static com.xebialabs.deployit.checks.Checks.checkArgument;
import static com.xebialabs.deployit.checks.Checks.checkNotNull;
import static com.xebialabs.xlrelease.repository.Ids.isInRootFolder;
import static com.xebialabs.xlrelease.repository.Ids.isReleaseId;
import static java.util.stream.Collectors.joining;
import static java.util.stream.Collectors.toList;

@Component
public class TeamFacade {

    private PermissionChecker permissions;
    private TeamService teamService;
    private TeamViewConverter teamViewConverter;
    private TeamMemberViewConverter teamMemberViewConverter;
    private ReleaseActorService releaseActorService;
    private static final List<String> SYSTEM_TEAMS = Arrays.asList(Team.FOLDER_OWNER_TEAMNAME, Team.RELEASE_ADMIN_TEAMNAME, Team.TEMPLATE_OWNER_TEAMNAME);

    @Autowired
    public TeamFacade(PermissionChecker permissions,
                      TeamService teamService,
                      TeamViewConverter teamViewConverter,
                      TeamMemberViewConverter teamMemberViewConverter,
                      ReleaseActorService releaseActorService) {
        this.permissions = permissions;
        this.teamService = teamService;
        this.teamViewConverter = teamViewConverter;
        this.teamMemberViewConverter = teamMemberViewConverter;
        this.releaseActorService = releaseActorService;
    }

    public List<TeamView> getTeams(String containerId) {
        permissions.checkViewTeams(containerId);

        return teamService.getEffectiveTeams(containerId).stream()
                .map(team -> teamViewConverter.toView(team))
                .collect(toList());
    }

    public List<TeamView> setTeams(String containerId, List<TeamView> teamsToUpdate) {
        checkNotNull(teamsToUpdate, "No teams submitted for update.");
        List<com.xebialabs.xlrelease.views.TeamView> otherTeamViews = teamsToUpdate.stream()
                .map(teamView -> new com.xebialabs.xlrelease.views.TeamView(teamViewConverter.toTeam(teamView), teamMemberViewConverter))
                .collect(toList());
        permissions.checkEditTeams(containerId, otherTeamViews);
        checkPermissions(teamsToUpdate);

        List<Team> submittedTeams = teamsToUpdate.stream().map(teamView -> teamViewConverter.toTeam(teamView)).collect(toList());

        List<Team> updatedTeams;
        if (isReleaseId(containerId)) {
            updatedTeams = releaseActorService.updateTeams(containerId, submittedTeams);
        } else {
            if (isInRootFolder(containerId) || teamsToUpdate.size() > 0) {
                checkSystemTeamsArePresent(teamsToUpdate);
            }
            updatedTeams = this.teamService.saveTeamsToPlatform(containerId, submittedTeams);
        }
        return updatedTeams.stream().map(team -> teamViewConverter.toView(team)).collect(toList());
    }

    public void checkPermissions(final List<TeamView> teamsToUpdate) {
        Set<String> unknownPermissions = new TreeSet<>();
        teamsToUpdate.forEach(t -> unknownPermissions.addAll(t.getPermissions().stream().filter(p -> Permission.find(p) == null).collect(toList())));
        checkArgument(unknownPermissions.isEmpty(), "Unknown permissions found: '%s'", unknownPermissions.stream().collect(joining(", ")));
    }

    private void checkSystemTeamsArePresent(final List<TeamView> teamsToUpdate) {
        List<String> missingSystemTeams = new ArrayList<>(SYSTEM_TEAMS);
        List<String> updateTeams = teamsToUpdate.stream().map(TeamView::getTeamName).collect(toList());
        missingSystemTeams.removeAll(updateTeams);
        checkArgument(missingSystemTeams.isEmpty(), "Cannot update teams with the following system teams missing: %s", missingSystemTeams);
    }
}
