package com.xebialabs.xlrelease.domain.variables;

import java.util.Collection;
import java.util.Collections;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import com.xebialabs.deployit.booter.local.utils.Strings;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.xlrelease.repository.Ids;
import com.xebialabs.xlrelease.service.FolderVariableService;
import com.xebialabs.xlrelease.service.VariableService;

import static com.xebialabs.xlrelease.variable.VariableHelper.formatVariableIfNeeded;
import static com.xebialabs.xlrelease.variable.VariableHelper.isFolderVariable;
import static com.xebialabs.xlrelease.variable.VariableHelper.isGlobalVariable;
import static com.xebialabs.xlrelease.variable.VariableHelper.withoutVariableSyntax;
import static java.lang.String.format;

@Component
public class ListOfStringValueProvider implements ValueProvider<String, ListOfStringValueProviderConfiguration> {

    private VariableService variableService;

    private FolderVariableService folderVariableService;

    @Autowired
    public ListOfStringValueProvider(final VariableService variableService, final FolderVariableService folderVariableService) {
        this.variableService = variableService;
        this.folderVariableService = folderVariableService;
    }

    @Override
    public Class<ListOfStringValueProviderConfiguration> getValueProviderConfigurationClass() {
        return ListOfStringValueProviderConfiguration.class;
    }

    @Override
    public Collection<String> possibleValues(final ListOfStringValueProviderConfiguration config) {
        if (Strings.isBlank(config.getVariableKey())) {
            return config.getValues();
        } else {
            String key = withoutVariableSyntax(config.getVariableKey());
            if (isGlobalVariable(key)) {
                return getGlobalVariableValue(key, config);
            } else if (isFolderVariable(key)) {
                return getFolderVariableValue(key, config);
            } else {
                return getReleaseVariableValue(key, config);
            }
        }
    }

    private Collection<String> getReleaseVariableValue(String key, ListOfStringValueProviderConfiguration config) {
        String releaseId = Ids.releaseIdFrom(config.getId());
        Variable variable = variableService.findByKey(key, releaseId);
        return ensureCollectionAndGetValue(variable, config, true, key);
    }

    private Collection<String> getGlobalVariableValue(String key, ListOfStringValueProviderConfiguration config) {
        Variable global = variableService.findGlobalVariablesOrEmpty().getVariablesByKeys().get(key);
        return ensureCollectionAndGetValue(global, config, false, key);
    }

    private Collection<String> getFolderVariableValue(String key, ListOfStringValueProviderConfiguration config) {
        String folderId = Ids.findFolderId(config.getId());
        final Variable variable = folderVariableService.findByKey(key, folderId);
        return ensureCollectionAndGetValue(variable, config, false, key);
    }

    private Collection<String> ensureCollectionAndGetValue(Variable variable, ListOfStringValueProviderConfiguration config,
                                                           boolean failOnNull, String logVariableKey) {
        String listBoxKey = config.getVariable() == null ? null : config.getVariable().getKey();
        String releaseId = Ids.releaseIdFrom(config.getId());
        if (variable == null && failOnNull) {
            throw new IllegalArgumentException(format(
                    "Could not find variable %s in release %s to provide values for list box variable %s",
                    formatVariableIfNeeded(logVariableKey), releaseId, formatVariableIfNeeded(listBoxKey)));
        }
        if (variable != null && !Type.valueOf(ListStringVariable.class).equals(variable.getType()) &&
                !Type.valueOf(SetStringVariable.class).equals(variable.getType())) {
            throw new IllegalArgumentException(format(
                    "The variable %s specified in list box variable %s of release %s is not a list or set variable",
                    formatVariableIfNeeded(logVariableKey), formatVariableIfNeeded(listBoxKey), releaseId));
        }
        @SuppressWarnings("unchecked")
        Collection<String> variableValue = variable == null ? null : (Collection<String>) variable.getValue();
        if (variableValue == null) {
            return Collections.emptyList();
        }
        return variableValue;
    }
}
