package com.xebialabs.xlrelease.scheduler

import com.xebialabs.xlrelease.scheduler.domain.LocalJobRunner
import com.xebialabs.xlrelease.scheduler.workers.CompositeWorker
import grizzled.slf4j.Logging

import java.util.concurrent.atomic.AtomicBoolean

class WorkerThread(jobQueue: JobQueue, compositeWorker: CompositeWorker) extends Runnable with Logging {
  private val isRunning = new AtomicBoolean(true)

  def stop(): Unit = {
    isRunning.set(false)
    jobQueue.submit(StopWorkerThread())
  }

  override def run(): Unit = {
    while (isRunning.get()) {
      try {
        val possibleJob = jobQueue.get(LocalJobRunner.getId())
        possibleJob.foreach {
          case StopWorkerThread() =>
            logger.debug("Thread is stopped because stop job was received")
            isRunning.set(false)
            ()
          case job =>
            val isConfirmed = jobQueue.confirm(runnerId = LocalJobRunner.getId(), job.getId())
            if (isConfirmed) {
              try {
                compositeWorker.execute(job)
              } catch {
                case e: Exception =>
                  val taskJobDetail = if (job.isInstanceOf[TaskJob[_]]) s" of task ${job.asInstanceOf[TaskJob[_]].taskId}" else ""
                  logger.error(s"Unhandled exception while executing ${job.jobType} ${job.id}$taskJobDetail. Please report a defect.", e)
              } finally {
                jobQueue.finish(job.id)
              }
            } else {
              logger.warn(s"Job [$job] execution could not be confirmed. Execution was skipped.")
            }
        }
      } catch {
        case _: JobQueueStopped =>
          logger.info("Thread is being stopped because job queue is stopped")
          isRunning.set(false)
        case _: InterruptedException =>
          logger.info("Thread is being terminated by a shutdown")
          isRunning.set(false)
        case e: Exception => logger.error(s"Unhandled exception", e)
      }
    }
  }
}
