package com.xebialabs.xlrelease.script;

import java.util.*;
import java.util.function.Function;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import com.xebialabs.deployit.util.PasswordEncrypter;
import com.xebialabs.xlrelease.domain.Changes;
import com.xebialabs.xlrelease.domain.Release;
import com.xebialabs.xlrelease.domain.Task;
import com.xebialabs.xlrelease.domain.variables.FolderVariables;
import com.xebialabs.xlrelease.domain.variables.PasswordStringVariable;
import com.xebialabs.xlrelease.domain.variables.Variable;
import com.xebialabs.xlrelease.repository.Ids;
import com.xebialabs.xlrelease.security.PermissionChecker;
import com.xebialabs.xlrelease.service.CiIdService;
import com.xebialabs.xlrelease.service.FolderVariableService;
import com.xebialabs.xlrelease.service.VariableService;

import static com.xebialabs.xlrelease.domain.Changes.VariablesChanges;
import static com.xebialabs.xlrelease.repository.CiCloneHelper.cloneCi;
import static com.xebialabs.xlrelease.script.ScriptVariableProcessorFactory.folderVariablesProcessor;
import static com.xebialabs.xlrelease.script.ScriptVariableProcessorFactory.globalVariablesProcessor;
import static com.xebialabs.xlrelease.script.ScriptVariableProcessorFactory.releaseVariableProcessor;
import static com.xebialabs.xlrelease.security.XLReleasePermissions.EDIT_FOLDER_VARIABLES;
import static com.xebialabs.xlrelease.security.XLReleasePermissions.EDIT_GLOBAL_VARIABLES;
import static com.xebialabs.xlrelease.security.XLReleasePermissions.EDIT_RELEASE;
import static com.xebialabs.xlrelease.utils.CiHelper.eraseTokens;
import static com.xebialabs.xlrelease.utils.Collectors.toMap;

@Component
public class ScriptVariables {
    private VariableService variableService;
    private FolderVariableService folderVariableService;
    private PasswordEncrypter passwordEncrypter;
    private CiIdService ciIdService;
    private PermissionChecker permissionChecker;

    @Autowired
    public ScriptVariables(VariableService variableService, FolderVariableService folderVariableService, PasswordEncrypter passwordEncrypter,
                           CiIdService ciIdService, PermissionChecker permissionChecker) {
        this.variableService = variableService;
        this.folderVariableService = folderVariableService;
        this.passwordEncrypter = passwordEncrypter;
        this.ciIdService = ciIdService;
        this.permissionChecker = permissionChecker;
    }

    public VariablesHolderForScriptContext createVariablesHolderForScriptContext(Release release,
                                                                                 String folderId,
                                                                                 Function<DefaultScriptService.VariablesUpdateHolder, DefaultScriptService.ScriptTaskResults> variablesSynchronizationCallback) {
        Map<String, Variable> initialReleaseVariables = initialReleaseVariables(release);
        Map<String, Variable> initialFolderVariables  = initialFolderVariables(folderId);
        Map<String, Variable> initialGlobalVariables  = initialGlobalVariables();
        return new VariablesHolderForScriptContext(
                release,
                folderId,
                initialReleaseVariables, initialFolderVariables, initialGlobalVariables,
                variablesSynchronizationCallback
        );
    }

    public VariablesHolderForScriptContext createVariablesHolderForScriptContext(Release release, String folderId) {
        return createVariablesHolderForScriptContext(release, folderId, noVariablesSynchronizationCallback());
    }

    private Function<DefaultScriptService.VariablesUpdateHolder, DefaultScriptService.ScriptTaskResults> noVariablesSynchronizationCallback() {
        return v -> new DefaultScriptService.ScriptTaskResults(
                Changes.VariablesChanges.EMPTY,
                Changes.VariablesChanges.EMPTY,
                Changes.VariablesChanges.EMPTY
        );
    }

    private Map<String, Variable> initialReleaseVariables(Release release) {
        return encryptAndNormalize(cloneReleaseVariables(release));
    }

    public Map<String, Variable> initialGlobalVariables() {
        return encryptAndNormalize(cloneGlobalVariables());
    }

    public Map<String, Variable> initialFolderVariables(String folderId) {
        return encryptAndNormalize(cloneFolderVariables(folderId));
    }

    public XlrScriptVariables asXlrScriptVariables(VariablesHolderForScriptContext variablesHolderForScriptContext) {
        Map<String, Object> folderVariables = folderVariables(variablesHolderForScriptContext);
        Map<String, Object> globalVariables = globalVariables(variablesHolderForScriptContext);
        Map<String, Object> releaseVariables = releaseVariables(variablesHolderForScriptContext);
        return new XlrScriptVariables(globalVariables, folderVariables, releaseVariables);
    }

    private Map<String, Object> releaseVariables(VariablesHolderForScriptContext variablesHolderForScriptContext) {
        Map<String, Object> releaseVariables = toVariableValues(variablesHolderForScriptContext.getPreviousReleaseVariables());
        Release release = variablesHolderForScriptContext.getRelease();
        if (permissionChecker.isNotAuthenticated() || !permissionChecker.hasPermission(EDIT_RELEASE, release)) {
            releaseVariables = Collections.unmodifiableMap(releaseVariables);
        }
        return releaseVariables;
    }

    private Map<String, Object> globalVariables(VariablesHolderForScriptContext variablesHolderForScriptContext) {
        Map<String, Object> globalVariables = toVariableValues(variablesHolderForScriptContext.getPreviousGlobalVariables());
        if (permissionChecker.isNotAuthenticated() || !permissionChecker.hasGlobalPermission(EDIT_GLOBAL_VARIABLES)) {
            globalVariables = Collections.unmodifiableMap(globalVariables);
        }
        return globalVariables;
    }

    private Map<String, Object> folderVariables(VariablesHolderForScriptContext variablesHolderForScriptContext) {
        Map<String, Object> folderVariables = toVariableValues(variablesHolderForScriptContext.getPreviousFolderVariables());
        String folderId = variablesHolderForScriptContext.getFolderId();
        if (permissionChecker.isNotAuthenticated() || !Ids.isFolderId(folderId) || !permissionChecker.hasPermission(EDIT_FOLDER_VARIABLES, folderId)) {
            folderVariables = Collections.unmodifiableMap(folderVariables);
        }
        return  folderVariables;
    }

    public VariablesChanges detectReleaseVariablesChanges(Task task, DefaultScriptService.VariablesUpdateHolder variableHolder) {
        Map<String, Object> releaseVariables = variableHolder.getReleaseVariables();
        Map<String, Variable> previousReleaseVariables = variableHolder.getInitialReleaseVariables();
        return releaseVariableProcessor(ciIdService, task.getRelease())
                .processVariablesAndUpdateInContainer(releaseVariables, previousReleaseVariables);
    }

    public VariablesChanges detectGlobalVariablesChanges(Task task, DefaultScriptService.VariablesUpdateHolder variableHolder) {
        Map<String, Object> globalVariables = variableHolder.getGlobalVariables();
        Map<String, Variable> previousGlobalVariables = variableHolder.getInitialGlobalVariables();
        return globalVariablesProcessor(ciIdService, task, variableService.findGlobalVariablesOrEmpty())
                .processVariablesAndUpdateInContainer(globalVariables, previousGlobalVariables);
    }

    public VariablesChanges detectFolderVariablesChanges(Task task, DefaultScriptService.VariablesUpdateHolder variableHolder) {
        Map<String, Object> scriptFolderVariables = variableHolder.getFolderVariables();
        Map<String, Variable> previousFolderVariables = variableHolder.getInitialFolderVariables();
        final FolderVariables folderVariables = folderVariableService.getAllFromAncestry(task.getRelease().findFolderId());
        return folderVariablesProcessor(permissionChecker, task, folderVariables)
                .processVariablesAndUpdateInContainer(scriptFolderVariables, previousFolderVariables);
    }

    public void processReleaseVariablesChanges(final Release release, final VariablesChanges releaseVariablesChanges) {
        List<Variable> updatedList = new ArrayList<>(release.getVariables());
        boolean variablesUpdated = !releaseVariablesChanges.getUpdatedVariables().isEmpty();
        boolean variablesRemoved = updatedList.removeAll(releaseVariablesChanges.getDeletedVariables());
        releaseVariablesChanges.getUpdatedVariables().forEach(updated -> {
            int index = updatedList.indexOf(updated);
            if (index >= 0) {
                updatedList.set(index, updated);
            }
        });
        boolean variablesAdded = updatedList.addAll(releaseVariablesChanges.getCreatedVariables());
        if (variablesUpdated || variablesAdded || variablesRemoved) {
            variableService.updateReleaseVariables(release, updatedList);
        }
    }

    public void processGlobalVariablesChanges(final VariablesChanges globalVariablesChanges) {
        globalVariablesChanges.getDeletedVariables().forEach(v ->
                variableService.deleteGlobalVariable(v.getId()));
        globalVariablesChanges.getCreatedVariables().forEach(
                variableService::addGlobalVariable);
        globalVariablesChanges.getUpdatedVariables().forEach(
                variableService::updateGlobalVariable);
    }


    public void processFolderVariablesChanges(final VariablesChanges folderVariablesChanges) {
        folderVariablesChanges.getDeletedVariables().forEach(variable -> folderVariableService.deleteFolderVariable(variable.getId()));
        folderVariablesChanges.getCreatedVariables().forEach(variable -> folderVariableService.createFolderVariable(variable));
        folderVariablesChanges.getUpdatedVariables().forEach(variable -> folderVariableService.updateFolderVariable(variable));
    }

    private boolean isExternalValueVariable(Variable variable) {
        return variable != null && variable.isPassword()
                && ((PasswordStringVariable)variable).getExternalVariableValue() != null;
    }

    private void removeExternalValueVariables(Map<String, Variable> variables) {
        if (variables == null || variables.isEmpty()) {
            return;
        }

        variables.entrySet().removeIf(entry -> isExternalValueVariable(entry.getValue()));
    }

    private Map<String, Variable> cloneGlobalVariables() {
        Map<String, Variable> previousGlobalVariables = variableService.findGlobalVariablesOrEmpty().getVariablesByKeys();
        removeExternalValueVariables(previousGlobalVariables);
        cloneVariableValue(previousGlobalVariables);
        return previousGlobalVariables;
    }

    private Map<String, Variable> cloneFolderVariables(String folderId) {
        final FolderVariables folderVariables = folderVariableService.getAllFromAncestry(folderId);
        Map<String, Variable> previousFolderVariables = folderVariables.getVariablesByKeys();
        removeExternalValueVariables(previousFolderVariables);
        cloneVariableValue(previousFolderVariables);
        return previousFolderVariables;
    }

    private Map<String, Variable> cloneReleaseVariables(Release release) {
        Map<String, Variable> previousReleaseVariables = release.getVariablesByKeys();
        removeExternalValueVariables(previousReleaseVariables);
        cloneVariableValue(previousReleaseVariables);
        return previousReleaseVariables;
    }

    private void cloneVariableValue(final Map<String, Variable> variablesMap) {
        variablesMap.entrySet().forEach(entry -> {
            eraseTokens(entry.getValue());
            entry.setValue(cloneCi(entry.getValue()));
        });
    }

    private Map<String, Variable> encryptAndNormalize(Map<String, Variable> variables) {
        return variables.entrySet().stream()
                .map(this::encryptPasswords)
                .map(this::normalizeEmptyValues)
                .collect(toMap(Map.Entry::getKey, Map.Entry::getValue));
    }

    public Map<String, Object> toVariableValues(Map<String, Variable> variables) {
        return encryptAndNormalize(variables).entrySet().stream()
                .collect(toMap(Map.Entry::getKey, entry -> entry.getValue().getValue()));
    }

    private Map.Entry<String, Variable> encryptPasswords(Map.Entry<String, Variable> entry) {
        entry.setValue(encryptPassword(entry.getValue()));
        return entry;
    }

    private Map.Entry<String, Variable> normalizeEmptyValues(Map.Entry<String, Variable> entry) {
        Variable v = entry.getValue();
        v.setUntypedValue(Optional.ofNullable(v.getValue()).orElse(v.getEmptyValue()));
        return entry;
    }

    private Variable encryptPassword(Variable variable) {
        if (variable.isPassword()) {
            if (!variable.isValueEmpty() && !passwordEncrypter.isEncrypted(variable.getValueAsString())) {
                variable.setUntypedValue(passwordEncrypter.encrypt(variable.getValueAsString()));
            }
        }
        return variable;
    }
}
