package com.xebialabs.xlrelease.service

import com.codahale.metrics.annotation.Timed
import com.xebialabs.deployit.booter.local.utils.Strings.isNotBlank
import com.xebialabs.xlrelease.domain.UserToken
import com.xebialabs.xlrelease.domain.events.{UserTokenAboutToExpireEvent, UserTokenCreatedEvent, UserTokenDeletedEvent, UserTokensDeletedEvent}
import com.xebialabs.xlrelease.events.EventBus
import com.xebialabs.xlrelease.repository.UserTokenRepository.{ByUsername, TokensAboutToExpire}
import com.xebialabs.xlrelease.repository.{CiCloneHelper, UserTokenRepository}
import com.xebialabs.xlrelease.utils.{PasswordVerificationUtils, TokenGenerator, TokenTypes}
import grizzled.slf4j.Logging
import org.springframework.data.domain.{Page, PageRequest, Pageable}

import java.time.{Instant, LocalTime, ZoneId}
import java.util.Date
import scala.jdk.CollectionConverters._

class UserTokenService(eventBus: EventBus, userTokenRepository: UserTokenRepository) extends Logging {

  @Timed
  def createUserToken(username: String, tokenNote: String, expiryDate: Date): UserToken = {
    val token: String = TokenGenerator.generate(TokenTypes.RPA)
    val userToken: UserToken = new UserToken
    userToken.tokenNote = tokenNote
    userToken.username = username
    userToken.createdDate = new Date()
    if (null == expiryDate) {
      userToken.expiryDate = expiryDate
    } else {
      // set expiry date as end of day
      val localExpiryDate = expiryDate.toInstant.atZone(ZoneId.systemDefault()).toLocalDate.atTime(LocalTime.MAX)
      userToken.expiryDate = Date.from(localExpiryDate.atZone(ZoneId.systemDefault()).toInstant)
    }

    val ciUid = userTokenRepository.create(userToken, TokenGenerator.hash(token))
    userToken.ciUid = ciUid
    userToken.token = token

    publishTokenCreatedEvent(userToken)
    userToken
  }

  @Timed
  def deleteUserToken(userToken: UserToken): Unit = {
    userTokenRepository.delete(userToken.ciUid)
    eventBus.publish(UserTokenDeletedEvent(userToken))
  }

  @Timed
  def deleteAllUserToken(username: String): Unit = {
    if (isNotBlank(username)) {
      userTokenRepository.delete(username: String)
      eventBus.publish(UserTokensDeletedEvent(username))
    } else {
      throw new IllegalArgumentException("Username can't be blank")
    }
  }

  @Timed
  def find(username: String, pageable: Pageable): Page[UserToken] = {
    userTokenRepository.query(ByUsername(username), pageable)
  }

  @Timed
  def findByCiUid(ciUid: Integer): Option[UserToken] = {
    userTokenRepository.findByCiUid(ciUid)
  }

  @Timed
  def findByUserToken(tokenHash: String): Option[UserToken] = {
    userTokenRepository.findByUserToken(tokenHash)
  }

  @Timed
  def notifyTokensAboutToExpire(): Unit = {
    val localTime = Instant.now().atZone(ZoneId.systemDefault).toLocalDate.atTime(LocalTime.MAX).plusDays(1)
    val expiryDate = Date.from(localTime.atZone(ZoneId.systemDefault()).toInstant)
    val tokensAboutToExpire = userTokenRepository.query(TokensAboutToExpire(expiryDate), PageRequest.ofSize(1024)).getContent.asScala

    tokensAboutToExpire.foreach { userToken =>
      eventBus.publish(UserTokenAboutToExpireEvent(userToken))
      userTokenRepository.updateTokenExpiredNotified(userToken.ciUid)
    }
  }



  @Timed
  def userTokenExists(username: String, tokenNote: String): Boolean = {
    userTokenRepository.findByUserAndNote(username, tokenNote) match {
      case Some(_) =>
        logger.debug(s"Personal access token was found with user[$username] and note[$tokenNote]")
        true
      case None =>
        logger.debug(s"Personal access token was not found with user[$username] and note[$tokenNote]")
        false
    }
  }

  private def publishTokenCreatedEvent(userToken: UserToken): Unit = {
    val userTokenForEvent = CiCloneHelper.cloneCi(userToken)
    userTokenForEvent.token = PasswordVerificationUtils.PASSWORD_MASK //Don't leak token value in events
    eventBus.publish(UserTokenCreatedEvent(userTokenForEvent))
  }
}
