package com.xebialabs.xlrelease.upgrade.json

import com.xebialabs.deployit.server.api.upgrade.Version
import com.xebialabs.deployit.util.PasswordEncrypter
import com.xebialabs.xlrelease.domain.variables.reference.ReleasePropertyVariableKey.isReleasePropertyVariableKey
import com.xebialabs.xlrelease.upgrade.Components.XL_RELEASE_COMPONENT
import com.xebialabs.xlrelease.upgrade.{JsonUpgrade, UpgradeResult}
import com.xebialabs.xlrelease.variable.VariableHelper._
import org.codehaus.jettison.json.{JSONArray, JSONObject}
import org.springframework.stereotype.Component

import java.util.regex.Pattern
import scala.collection.mutable
import scala.jdk.CollectionConverters._
import scala.util.Try

@Component
class XLRelease480RenameReservedVariablesJson extends JsonUpgrade {

  override def upgradeVersion(): Version = Version.valueOf(XL_RELEASE_COMPONENT, "4.8.0#7")

  override def performUpgrade(releaseJson: JSONObject): UpgradeResult = {

    replaceReservedVariablesRecursively(releaseJson)

    val replacedKeys = new mutable.HashSet[String]()
    replacedKeys ++= replaceInVariablesArray(releaseJson)
    replacedKeys ++= replaceInVariablesMap(releaseJson, "variableValues")
    replacedKeys ++= replaceInVariablesMap(releaseJson, "passwordVariableValues")
    replacedKeys ++= replaceInTriggers(releaseJson)

    val warnings = replacedKeys.toList.map { key =>
      val oldKey = withVariableSyntax(key)
      val newKey = withVariableSyntax("_" + key)
      s"Replaced variable $oldKey with $newKey. Note that statements " +
        s"such as releaseVariables['$key'] in script tasks in this template will no longer work"
    }

    new UpgradeResult(true, warnings.asJava)
  }

  private val maybeReservedVariablePattern = Pattern.compile("\\$\\{(\\w+\\.[^\\}]*)\\}", Pattern.CASE_INSENSITIVE)

  private def replaceInVariablesArray(releaseJson: JSONObject): Seq[String] = {
    val releaseId = releaseJson.getString("id")
    Option(releaseJson.optJSONArray("variables")).map { variables =>
      (0 until variables.length())
        .flatMap { i =>
          Option(variables.optJSONObject(i)).flatMap(replaceVariableKey(_, releaseId))
        }
    }.getOrElse(Seq.empty)
  }

  private def replaceInTriggers(releaseJson: JSONObject): Seq[String] = {
    val triggers: Seq[JSONObject] = Option(releaseJson.optJSONArray("releaseTriggers")).map { triggers =>
      (0 until triggers.length()).map { i =>
        triggers.getJSONObject(i)
      }
    }.getOrElse(Seq())
    triggers.flatMap { triggerJson =>
      replaceInVariablesMap(triggerJson, "templateVariables") ++
        replaceInVariablesMap(triggerJson, "templatePasswordVariables")
    }
  }

  private def replaceInVariablesMap(jsonObject: JSONObject, property: String): Seq[String] = {
    Option(jsonObject.optJSONObject(property)).map(replaceInVariablesMap).getOrElse(Seq())
  }

  private def replaceInVariablesMap(valuesMap: JSONObject): Seq[String] = {
    val keysToReplace = valuesMap.keys().asScala.flatMap {
      case key: String if keyNeedsRenaming(key) => Some(key)
      case _ => None
    }.toList
    keysToReplace.foreach { key =>
      val newKey = if (containsOnlyVariable(key)) {
        withVariableSyntax("_" + withoutVariableSyntax(key))
      }
      else {
        "_" + key
      }
      val valueAtKey = valuesMap.get(key)
      valuesMap.put(newKey, valueAtKey)
      valuesMap.remove(key)
    }

    val keysIterator = valuesMap.keys()
    keysIterator.forEachRemaining {
      case key: String if isReleasePropertyVariableKey(key) => keysIterator.remove()
      case _ =>
    }

    keysToReplace.map(withoutVariableSyntax)
  }

  private def replaceVariableKey(variable: JSONObject, releaseId: String): Option[String] = {
    val name = variable.getString("key")
    if (isGlobalVariable(name)) {
      val newName = "_" + name
      variable.put("key", newName)
      Some(name)
    } else {
      None
    }
  }

  private def replaceReservedVariablesRecursively(jsonObject: JSONObject): JSONObject = {
    jsonObject.keys().forEachRemaining {
      case stringKey: String =>
        val replaced = replaceReserved(jsonObject.get(stringKey))
        jsonObject.put(stringKey, replaced)
      case _ =>
    }
    jsonObject
  }

  private[json] def replaceReserved(something: AnyRef): AnyRef = {
    something match {
      case obj: JSONObject => replaceReservedVariablesRecursively(obj)
      case value: String => replaceString(value)
      case array: JSONArray =>
        (0 until array.length()).foreach { i =>
          val item = array.get(i)
          array.put(i, replaceReserved(item))
        }
        array
      case other => other
    }
  }

  private def replaceString(value: String): String = {

    def replacePattern(text: String): String = {
      val matcher = maybeReservedVariablePattern.matcher(text)
      var keysToReplace = mutable.Set[String]()
      while (matcher.find()) {
        val key = matcher.group(1)
        if (keyNeedsRenaming(key)) {
          keysToReplace += key
        }
      }

      keysToReplace.foldLeft(text) { (replacedText, key) =>
        replacedText.replace("${" + key + "}", "${_" + key + "}")
      }
    }

    val encrypter = PasswordEncrypter.getInstance()
    if (encrypter.isEncoded(value)) {
      Try(encrypter.decrypt(value)).map { decrypted =>
        encrypter.encrypt(replacePattern(decrypted))
      }.getOrElse(value)
    } else {
      replacePattern(value)
    }

  }

  private def keyNeedsRenaming(keyMaybeFormatted: String) = {
    val key = formatVariableIfNeeded(keyMaybeFormatted)
    key.toLowerCase.startsWith("${global.") || (key.toLowerCase.startsWith("${release.") && !isReleasePropertyVariableKey(key))
  }

}
