package com.xebialabs.xlrelease.upgrade.json

import com.xebialabs.deployit.repository.RepositoryAdapter
import com.xebialabs.deployit.server.api.upgrade.Version
import com.xebialabs.xlrelease.builder.VariableBuilder._
import com.xebialabs.xlrelease.domain.variables.Variable
import com.xebialabs.xlrelease.serialization.json.utils.CiSerializerHelper.{newEncryptingConverter, serialize}
import com.xebialabs.xlrelease.service.{ArchivedReleaseDeserialization, CiIdService}
import com.xebialabs.xlrelease.upgrade.Components.XL_RELEASE_COMPONENT
import com.xebialabs.xlrelease.upgrade.{JsonUpgrade, UpgradeResult}
import com.xebialabs.xlrelease.variable.VariablePersistenceHelper.fixUpVariableIds
import org.codehaus.jettison.json.{JSONArray, JSONObject}
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

import java.util
import scala.jdk.CollectionConverters._
import scala.util.Try

@Component
class XLRelease480UpgradeVariablesJson @Autowired()(val repositoryAdapter: RepositoryAdapter,
                                                    ciIdService: CiIdService) extends JsonUpgrade
  with ArchivedReleaseDeserialization {

  override def upgradeVersion(): Version = Version.valueOf(XL_RELEASE_COMPONENT, "4.8.0#9")

  override def performUpgrade(releaseJson: JSONObject): UpgradeResult = {
    // Can't use empty maps here because we need to convert "variableValues: {}" too
    val variableValues = Try(releaseJson.getJSONObject("variableValues")).toOption.map(jsonToMap)
    val passwordVariables = Try(releaseJson.getJSONObject("passwordVariableValues")).toOption.map(jsonToMap)

    variableValues.foreach(_ => releaseJson.remove("variableValues"))
    passwordVariables.foreach(_ => releaseJson.remove("passwordVariableValues"))

    val allTriggerVars = Option(releaseJson.optJSONArray("releaseTriggers")).map { triggers =>
      (0 until triggers.length()).map(i => collectTriggerVariables(triggers.getJSONObject(i)))
    }

    if (isNotEmpty(variableValues) || isNotEmpty(passwordVariables) || isNotEmpty(allTriggerVars)) {
      val release = deserializeArchivedRelease(releaseJson.toString, repositoryAdapter, newEncryptingConverter())

      release.scanAndAddNewVariables()
      variableValues.foreach(values => release.setVariableValues(values.asJava))
      passwordVariables.foreach(values => release.setPasswordVariableValues(values.asJava))
      fixUpVariableIds(release.getId, release.getVariables, ciIdService)
      releaseJson.put("variables", new JSONObject(serialize(release)).getJSONArray("variables"))

      allTriggerVars.foreach { triggersVars =>
        (release.getReleaseTriggers.asScala zip triggersVars).foreach { case (trigger, vars) =>
          trigger.setVariables(vars.asJava)
          fixUpVariableIds(trigger.getId, trigger.getVariables, ciIdService)
        }
      }
      releaseJson.put("releaseTriggers", new JSONArray(serialize(release.getReleaseTriggers)))
    }
    new UpgradeResult(true, new util.ArrayList[String]())
  }

  private def collectTriggerVariables(trigger: JSONObject): List[Variable] = {
    val varsObj = Option(trigger.optJSONObject("templateVariables")).map(jsonToMap)
    val passVarsObj = Option(trigger.optJSONObject("templatePasswordVariables")).map(jsonToMap)

    varsObj.foreach(_ => trigger.remove("templateVariables"))
    passVarsObj.foreach(_ => trigger.remove("templatePasswordVariables"))

    val stringVars = varsObj.getOrElse(Map.empty[String, String]).map { case (k, v) => newStringVariable(k, v).build() }
    val passwordVars = passVarsObj.getOrElse(Map.empty[String, String]).map { case (k, v) => newPasswordStringVariable(k, v).build() }
    (stringVars ++ passwordVars).toList
  }

  private def jsonToMap(json: JSONObject): Map[String, String] =
    json.keys.asScala.map { key => key -> json.get(key.toString) }
      .collect { case (k: String, v: String) => k -> v }.toMap

  private def isNotEmpty[A <: Iterable[_]](iterableOpt: Option[A]): Boolean =
    iterableOpt.isDefined && iterableOpt.get.nonEmpty

}
