package com.xebialabs.xlrelease.actors.sharding
import akka.actor.{ActorRef, Props}
import akka.cluster.sharding.ShardCoordinator.LeastShardAllocationStrategy
import akka.cluster.sharding.{ClusterSharding, ClusterShardingSettings, ShardRegion}
import com.xebialabs.xlrelease.actors.sharding.ReleaseShardingMessages.{PrepareForBalance, ReleaseAction}
import com.xebialabs.xlrelease.actors.{ActorSystemHolder, _}
import com.xebialabs.xlrelease.config.XlrConfig
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired

/**
  * Messages accepted by sharding actor which forwards payload to the correct release.
  */
object ReleaseShardingMessages {

  case class ReleaseAction(releaseId: String, payload: AnyRef)

  case object PrepareForBalance

  case object ReleaseExecutorActorTerminated

}


class FullReleaseSharding @Autowired()(xlrConfig: XlrConfig, systemHolder: ActorSystemHolder) extends Logging {

  private val entityIdExtractor: ShardRegion.ExtractEntityId = {
    case ReleaseAction(id, msg) => releaseId2ActorName(id) -> msg
  }

  private val shardResolver: Int => ShardRegion.ExtractShardId = (numberOfShards: Int) => {
    case ReleaseAction(id, _) => releaseIdToShardId(id, numberOfShards)
    case ShardRegion.StartEntity(id) => releaseIdToShardId(id, numberOfShards)
  }

  private def releaseIdToShardId(releaseId: String, numberOfShards: Int) = (Math.abs(releaseId2ActorName(releaseId).hashCode) % numberOfShards).toString

  def createShardRegion(releaseActorProps: Props): ActorRef = {

    val actorSystem = systemHolder.actorSystem

    val numberOfReleaseShards = xlrConfig.sharding.numberOfReleaseShards

    val clusterSharding = ClusterSharding(actorSystem)

    val shardingSettings = ClusterShardingSettings(actorSystem)
    val rebalanceThreshold = shardingSettings.tuningParameters.leastShardAllocationRebalanceThreshold
    val simultaneousRebalance = shardingSettings.tuningParameters.leastShardAllocationMaxSimultaneousRebalance

    logger.debug(s"Setting up cluster sharding for ${actorSystem.name} with re-balance threshold = $rebalanceThreshold and simultaniousRebalance = $simultaneousRebalance")

    clusterSharding.start(
      typeName = "release",
      entityProps = releaseActorProps,
      settings = shardingSettings,
      extractEntityId = entityIdExtractor,
      extractShardId = shardResolver(numberOfReleaseShards),
      handOffStopMessage = PrepareForBalance,
      allocationStrategy = new LeastShardAllocationStrategy(rebalanceThreshold, simultaneousRebalance)
    )

  }
}
