package com.xebialabs.xlrelease.api.internal;

import java.io.IOException;
import java.util.Locale;
import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.Response;
import org.jboss.resteasy.annotations.cache.NoCache;
import org.springframework.core.io.Resource;
import org.springframework.stereotype.Controller;

import com.codahale.metrics.annotation.Timed;

import com.xebialabs.xlrelease.utils.ResourceUtils;

import static javax.ws.rs.core.Response.Status.NOT_FOUND;

/**
 * Static web resources (images, stylesheets...) used by the Digital.ai Release UI.
 */
@Path("/")
@Controller
public class XLReleaseStaticResource {
    public XLReleaseStaticResource() {
    }

    @GET
    @NoCache
    @Produces("text/html;charset=UTF-8")
    public Response redirectToRoot() throws IOException {
        return getRoot("index.html");
    }

    @GET
    @Timed
    @Path("{path : static/.*\\.html}")
    @NoCache
    @Produces("text/html;charset=UTF-8")
    public Response getRoot(@PathParam("path") String path) throws IOException {
        return streamFromWeb(path);
    }

    @GET
    @Timed
    @Path("{path : static/.*\\.otf}")
    @Produces("font/opentype")
    public Response getOtfFont(@PathParam("path") String path) throws IOException {
        return streamFromWeb(path);
    }

    @GET
    @Timed
    @Path("{path : static/.*\\.eot}")
    @Produces("application/vnd.ms-fontobject")
    public Response getEotFont(@PathParam("path") String path) throws IOException {
        return streamFromWeb(path);
    }

    @GET
    @Timed
    @Path("{path : .*\\.woff}")
    @Produces("application/font-woff")
    public Response getWoffFont(@PathParam("path") String path) throws IOException {
        return streamFromWeb(path);
    }

    @GET
    @Timed
    @Path("{path : .*\\.woff2}")
    @Produces("application/font-woff2")
    public Response getWoff2Font(@PathParam("path") String path) throws IOException {
        return streamFromWeb(path);
    }

    @GET
    @Timed
    @Path("{path : static/.*\\.ttf}")
    @Produces("application/x-font-ttf")
    public Response getTtfFont(@PathParam("path") String path) throws IOException {
        return streamFromWeb(path);
    }

    @GET
    @Timed
    @Path("{path : static/.*\\.svg}")
    @Produces("image/svg+xml")
    public Response getSvg(@PathParam("path") String path) throws IOException {
        return streamFromWebOrClasspathRoot(path);
    }

    @GET
    @Timed
    @Path("{path : static/.*\\.png}")
    @Produces("image/png")
    public Response getPng(@PathParam("path") String path) throws IOException {
        return streamFromWebOrClasspathRoot(path);
    }

    @GET
    @Timed
    @Path("{path : static/.*\\.gif}")
    @Produces("image/gif")
    public Response getGif(@PathParam("path") String path) throws IOException {
        return streamFromWebOrClasspathRoot(path);
    }


    @GET
    @Timed
    @Path("{path : static/.*\\.jpg}")
    @Produces("image/jpg")
    public Response getJpg(@PathParam("path") String path) throws IOException {
        return streamFromWebOrClasspathRoot(path);
    }

    @GET
    @Timed
    @Path("{path : static/.*\\.ico}")
    @Produces("image/x-icon")
    public Response getFavicon(@PathParam("path") String path) throws IOException {
        return streamFromWeb(path);
    }

    @GET
    @Timed
    @Path("{path : static/.*\\.map}")
    @Produces("application/json")
    public Response getJsMap(@PathParam("path") String path) throws IOException {
        return streamFromWeb(path);
    }

    @GET
    @Timed
    @Path("{path : static/.*\\.json}")
    @Produces("application/json")
    public Response getJson(@PathParam("path") String path) throws IOException {
        return streamFromWeb(path);
    }

    @GET
    @Timed
    @Path("{path : static/.*\\.js}")
    @Produces("application/javascript")
    public Response getJs(@PathParam("path") String path, @Context HttpServletRequest request) throws IOException {
        if (path.endsWith("locale.js")) {
            return getLocaleJs(request);
        }

        return streamFromWeb(path);
    }

    @GET
    @Timed
    @Path("{path : static/.*\\.css}")
    @Produces("text/css")
    public Response getCss(@PathParam("path") String path) throws IOException {
        return streamFromWeb(path);
    }

    private Response streamFromWeb(String path) throws IOException {
        return toResponse(ResourceUtils.webResource(removeStaticAndVersionPrefix(path)));
    }

    private Response streamFromClasspathRoot(String path) throws IOException {
        return toResponse(ResourceUtils.resource(removeStaticAndVersionPrefix(path)));
    }

    private Response streamFromWebOrClasspathRoot(String path) throws IOException {
        Response response = streamFromWeb(path);
        if (response.getStatus() == NOT_FOUND.getStatusCode()) {
            // in case a custom script task icon is queried, for backwards compatibility
            response = streamFromClasspathRoot(path);
        }
        return response;
    }

    /**
     * Given a path : static/2.2.1/img/logo.png it returns img/logo.png
     */
    private String removeStaticAndVersionPrefix(String path) {
        return path.replaceAll("^static/[^/]+/", "");
    }

    private Response getLocaleJs(@Context HttpServletRequest request) throws IOException {
        Response response = getAngularLocaleResource(request.getLocale());
        if (response.getStatus() == NOT_FOUND.getStatusCode()) {
            response = getAngularLocaleResource(Locale.ENGLISH);
        }
        return response;
    }

    private Response getAngularLocaleResource(Locale locale) throws IOException {
        String localeName = locale.toString();
        String localeNameForAngular = localeName.toLowerCase().replace("_", "-");
        String path = "components/angular-i18n/angular-locale_" + localeNameForAngular + ".js";

        return streamFromWeb(path);
    }

    private Response toResponse(final Resource resource) throws IOException {
        if (!resource.exists()) {
            return Response.status(NOT_FOUND).build();
        }
        return Response.ok(resource.getInputStream()).build();
    }

}
