package com.xebialabs.xlrelease.api.v1.impl;

import java.util.List;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import com.codahale.metrics.annotation.Timed;

import com.xebialabs.deployit.checks.Checks;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.xlrelease.api.v1.TaskReportingApi;
import com.xebialabs.xlrelease.domain.facet.Facet;
import com.xebialabs.xlrelease.domain.facet.TaskReportingRecord;
import com.xebialabs.xlrelease.domain.udm.reporting.*;
import com.xebialabs.xlrelease.security.FacetPermissionChecker;
import com.xebialabs.xlrelease.security.PermissionChecker;
import com.xebialabs.xlrelease.service.FacetService;

import static com.xebialabs.xlrelease.repository.Ids.releaseIdFrom;

// This API only be exported to plugin scripts and not through REST
@Component
public class TaskReportingApiImpl implements TaskReportingApi {

    private final FacetService facetService;
    private final PermissionChecker permissions;
    private final FacetPermissionChecker facetPermissions;

    @Autowired
    public TaskReportingApiImpl(FacetService facetService, PermissionChecker permissions, FacetPermissionChecker facetPermissions) {
        Checks.checkNotNull(facetService, "FacetService must be provided. If you're testing think twice as this object will have impact on other tests!");
        Checks.checkNotNull(permissions, "PermissionChecker must be provided");
        Checks.checkNotNull(facetPermissions, "FacetPermissionChecker must be provided");
        this.facetService = facetService;
        this.permissions = permissions;
        this.facetPermissions = facetPermissions;
    }

    @Timed
    @Override
    public List<TaskReportingRecord> addRecord(TaskReportingRecord record, boolean applyTaskAttributes) {
        record.setCreatedViaApi(true);
        performChecksAndSetRetryCounter(record);

        return facetService.addTaskReportingRecord(record, applyTaskAttributes);
    }

    @Override
    public TaskReportingRecord addRecord(final TaskReportingRecord record) {
        return this.addRecord(record, false).get(0);
    }

    @Override
    public ItsmRecord newItsmRecord() {
        return newReportingRecord("udm.ItsmRecord");
    }

    @Override
    public PlanRecord newPlanRecord() {
        return newReportingRecord("udm.PlanRecord");
    }

    @Override
    public BuildRecord newBuildRecord() {
        return newReportingRecord("udm.BuildRecord");
    }

    @Override
    public DeploymentRecord newDeploymentRecord() {
        return newReportingRecord("udm.DeploymentRecord");
    }

    @Override
    public CodeComplianceRecord newCodeComplianceRecord() {
        return newReportingRecord("udm.CodeComplianceRecord");
    }

    private <F extends Facet> F newReportingRecord(String type) {
        return Type.valueOf(type).getDescriptor().newInstance("");
    }

    private void performChecksAndSetRetryCounter(TaskReportingRecord record) {
        if (record.getTargetId() == null || record.getTargetId().trim().isEmpty()) {
            // get task id from current context
            if (facetPermissions.hasScriptTask()) {
                record.setTargetId(facetPermissions.getScriptTask().getId());
            } else {
                throw new IllegalStateException("reporting record should have targetId populated unless it is called from within script task context");
            }
        }
        String releaseId = releaseIdFrom(record.getTargetId());
        facetPermissions.checkThatScriptTaskBelongsToReleaseOrOtherwise(releaseId, () -> {
            permissions.checkView(releaseId);
            permissions.checkEdit(releaseId);
            permissions.checkEditTask(releaseId);
        });
        if (facetPermissions.hasScriptTask()) {
            record.setRetryAttemptNumber(facetPermissions.getScriptTask().getFailuresCount());
        }
    }

}
