package com.xebialabs.xlrelease.scheduler.service

import com.codahale.metrics.annotation.Timed
import com.xebialabs.xlrelease.scheduler._
import com.xebialabs.xlrelease.scheduler.converters.TaskJobConverter
import com.xebialabs.xlrelease.scheduler.filters.JobFilters
import com.xebialabs.xlrelease.scheduler.repository._
import grizzled.slf4j.Logging
import org.springframework.data.domain.Pageable
import org.springframework.stereotype.Service

import scala.jdk.CollectionConverters._

@Service
class JobService(jobRepository: JobRepository, taskJobConverter: TaskJobConverter) extends Logging {

  @Timed
  def findAllTaskJobs(jobFilters: JobFilters, pageable: Pageable, ignoreUnknownType: Boolean): List[TaskJob[_]] = {
    val foundJobs = jobRepository.findAll(jobFilters, pageable).toList.asScala.toList
    foundJobs.map(jobRow => {
      try {
        taskJobConverter.fromJobRow(jobRow)
      } catch {
        case t: Throwable =>
          if (ignoreUnknownType) {
            logger.error(t.getMessage, t)
            null
          } else {
            throw t
          }
      }
    }).filterNot(_ == null)
  }

  @Timed
  def findByTaskId(taskId: String): Option[JobRow] = {
    jobRepository.findByTaskId(taskId)
  }

  @Timed
  def findDistinctNodeIds(): Set[String] = {
    jobRepository.findDistinctNodeIds()
  }

  @Timed
  def deleteByTaskId(taskId: String): Unit = {
    jobRepository.delete(DeleteByTaskId(taskId))
  }

  @Timed
  def deleteByTaskIdAndExecutionId(taskId: String, executionId: String): Unit = {
    jobRepository.delete(DeleteByTaskIdAndExecutionId(taskId, executionId))
  }

  @Timed
  def updateJobStatus(job: Job, jobStatus: JobStatus): Option[JobRow] = {
    jobRepository.update(UpdateJobStatus(JobRow(job), jobStatus)).toOption
  }

  @Timed
  def updateJobNodeAndStatus(job: Job, node: String, jobStatus: JobStatus): Option[JobRow] = {
    jobRepository.update(UpdateNodeAndStatus(JobRow(job), node, jobStatus)).toOption
  }

}
