package com.xebialabs.xlrelease.script

import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor
import com.xebialabs.deployit.plugin.api.reflect.PropertyKind._
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.deployit.util.PasswordEncrypter
import com.xebialabs.xlrelease.utils.CiHelper
import grizzled.slf4j.Logging
import org.springframework.util.StringUtils

import java.util
import scala.jdk.CollectionConverters._

object EncryptionHelper extends Logging {
  private val passwordEncrypter = PasswordEncrypter.getInstance()

  def decrypt(ci: ConfigurationItem): Unit = {
    CiHelper.getNestedCis(ci).forEach(decryptProperties)
    decryptDirectCiRefs(ci)
  }

  private def decryptDirectCiRefs(ci: ConfigurationItem): Unit = {
    ci.getType.getDescriptor.getPropertyDescriptors.asScala foreach {
      case pd if pd.getKind == CI => decryptProperties(pd.get(ci).asInstanceOf[ConfigurationItem])
      case pd if pd.getKind == SET_OF_CI => pd.get(ci).asInstanceOf[util.Set[ConfigurationItem]].forEach(decryptProperties)
      case pd if pd.getKind == LIST_OF_CI => pd.get(ci).asInstanceOf[util.List[ConfigurationItem]].forEach(decryptProperties)
      case _ =>
    }
  }

  private def decryptProperties(ci: ConfigurationItem): Unit = {
    if (null != ci) {
      for (pd <- ci.getType.getDescriptor.getPropertyDescriptors.asScala) {
        if (pd.isPassword) decrypt(ci, pd)
      }
    }
  }

  private def decrypt(ci: ConfigurationItem, pd: PropertyDescriptor): Unit = {
    logger.trace(s"Decrypting password property `${pd.getFqn}` of `${ci.getId}`")
    pd.getKind match {
      case SET_OF_STRING => pd.set(ci, decrypt(pd.get(ci).asInstanceOf[util.Set[String]]))
      case LIST_OF_STRING => pd.set(ci, decrypt(pd.get(ci).asInstanceOf[util.List[String]]))
      case MAP_STRING_STRING => pd.set(ci, decrypt(pd.get(ci).asInstanceOf[util.Map[String, String]]))
      case STRING => pd.set(ci, decrypt(pd.get(ci).asInstanceOf[String]))
      case _ => throw new IllegalStateException(s"Unable to decrypt password property '${pd.getFqn}' of kind '${pd.getKind}' on item '${ci.getId}'")
    }
  }

  private def decrypt(encryptedSet: util.Set[String]): util.Set[String] = (encryptedSet.asScala map decrypt).asJava

  private def decrypt(encryptedList: util.List[String]): util.List[String] = (encryptedList.asScala map decrypt).asJava

  private def decrypt(encryptedMap: util.Map[String, String]): util.Map[String, String] = (encryptedMap.asScala.view mapValues decrypt).toMap.asJava

  private def decrypt(passwordValue: String): String = {
    if (passwordValue != null) {
      passwordEncrypter.ensureDecrypted(passwordValue)
    } else {
      null
    }
  }

  def encrypt(ci: ConfigurationItem): Unit = {
    CiHelper.getNestedCis(ci).forEach(encryptProperties)
    encryptDirectCiRefs(ci)
  }

  private def encryptDirectCiRefs(ci: ConfigurationItem): Unit = {
    ci.getType.getDescriptor.getPropertyDescriptors.asScala foreach {
      case pd if pd.getKind == CI => encryptProperties(pd.get(ci).asInstanceOf[ConfigurationItem])
      case pd if pd.getKind == SET_OF_CI => pd.get(ci).asInstanceOf[util.Set[ConfigurationItem]].forEach(encryptProperties)
      case pd if pd.getKind == LIST_OF_CI => pd.get(ci).asInstanceOf[util.List[ConfigurationItem]].forEach(encryptProperties)
      case _ =>
    }
  }

  private def encryptProperties(ci: ConfigurationItem): Unit = {
    if (null != ci) {
      for (pd <- ci.getType.getDescriptor.getPropertyDescriptors.asScala) {
        if (pd.isPassword) encrypt(ci, pd)
      }
    }
  }

  private def encrypt(ci: ConfigurationItem, pd: PropertyDescriptor): Unit = {
    logger.trace(s"Encrypting password property `${pd.getFqn}` of `${ci.getId}`")
    pd.getKind match {
      case SET_OF_STRING => pd.set(ci, encrypt(pd.get(ci).asInstanceOf[util.Set[String]]))
      case LIST_OF_STRING => pd.set(ci, encrypt(pd.get(ci).asInstanceOf[util.List[String]]))
      case MAP_STRING_STRING => pd.set(ci, encrypt(pd.get(ci).asInstanceOf[util.Map[String, String]]))
      case STRING => pd.set(ci, encrypt(pd.get(ci).asInstanceOf[String]))
      case _ => throw new IllegalStateException(s"Unable to encrypt password property '${pd.getFqn}' of kind '${pd.getKind}' on item '${ci.getId}'")
    }
  }

  private def encrypt(decryptedSet: util.Set[String]): util.Set[String] = (decryptedSet.asScala map encrypt).asJava

  private def encrypt(decryptedList: util.List[String]): util.List[String] = (decryptedList.asScala map encrypt).asJava

  private def encrypt(decryptedMap: util.Map[String, String]): util.Map[String, String] = (decryptedMap.asScala.view mapValues encrypt).toMap.asJava

  private def encrypt(passwordValue: String): String = {
    if (StringUtils.hasText(passwordValue)) {
      passwordEncrypter.ensureEncrypted(passwordValue)
    } else {
      null
    }
  }
}
