package com.xebialabs.xlrelease.script;

import java.io.IOException;
import java.io.Writer;
import java.util.*;
import javax.script.ScriptContext;
import javax.script.ScriptException;
import org.python.core.PyException;
import org.python.core.PyType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plumbing.ExecutionOutputWriter;
import com.xebialabs.platform.script.jython.ThreadLocalWriterDecorator;
import com.xebialabs.xlrelease.domain.CustomScriptTask;
import com.xebialabs.xlrelease.domain.variables.StringVariable;
import com.xebialabs.xlrelease.domain.variables.Variable;
import com.xebialabs.xlrelease.utils.SensitiveValueScrubber;

import scala.Option;

import static com.xebialabs.xlrelease.variable.VariableFactory.createVariableByPropertyDescriptor;
import static java.lang.String.format;

public class ScriptServiceHelper {

    private static final Logger logger = LoggerFactory.getLogger(ScriptServiceHelper.class);

    @SuppressWarnings("unchecked")
    public static Map<String, Object> extractReleaseVariables(ScriptContext context) {
        Object attribute = context.getAttribute(XlrScriptContext.ATTRIBUTE_RELEASE_VARIABLES);
        if (attribute != null) {
            return (Map<String, Object>) attribute;
        } else {
            return Collections.emptyMap();
        }
    }

    @SuppressWarnings("unchecked")
    public static Map<String, Object> extractGlobalVariables(ScriptContext context) {
        Object attribute = context.getAttribute(XlrScriptContext.ATTRIBUTE_GLOBAL_VARIABLES);
        if (attribute != null) {
            return (Map<String, Object>) attribute;
        } else {
            return Collections.emptyMap();
        }
    }


    @SuppressWarnings("unchecked")
    public static Map<String, Object> extractFolderVariables(ScriptContext context) {
        Object attribute = context.getAttribute(XlrScriptContext.ATTRIBUTE_FOLDER_VARIABLES);
        if (attribute != null) {
            return (Map<String, Object>) attribute;
        } else {
            return Collections.emptyMap();
        }
    }

    public static Map<String, Object> extractTransitionalAndOutputPropertyValues(CustomScriptTask task,
                                                                                 ScriptContext scriptContext,
                                                                                 SensitiveValueScrubber scrubber) {
        final Collection<PropertyDescriptor> transitionalAndOutputProperties = task.getPythonScript().getTransitionalAndOutputProperties();
        return extractPropertyValues(task.getId(), transitionalAndOutputProperties, scriptContext, scrubber);
    }

    private static Map<String, Object> extractPropertyValues(String taskId,
                                                             Collection<PropertyDescriptor> propertyDescriptors,
                                                             ScriptContext scriptContext,
                                                             SensitiveValueScrubber scrubber) {
        final Map<String, Object> propertyValues = new HashMap<>();
        for (PropertyDescriptor propertyDescriptor : propertyDescriptors) {
            String propertyName = propertyDescriptor.getName();
            Object attr = scriptContext.getAttribute(propertyName);
            if (attr == null) {
                logger.debug("Task {} did not return value for property {}", taskId, propertyDescriptor.getName());
                continue;
            }

            attr = convertOutputPropertyValue(propertyDescriptor, attr, scrubber);
            propertyValues.put(propertyName, attr);
        }
        return propertyValues;
    }

    public static Map<String, Object> extractPropertyValues(String taskId,
                                                            Collection<PropertyDescriptor> propertyDescriptors,
                                                            Map<String, Object> outputAttributes,
                                                            SensitiveValueScrubber scrubber) {
        final Map<String, Object> propertyValues = new HashMap<>();
        for (PropertyDescriptor propertyDescriptor : propertyDescriptors) {
            String propertyName = propertyDescriptor.getName();
            Object attr = outputAttributes.get(propertyName);
            if (attr == null) {
                logger.debug("Task {} did not return value for property {}", taskId, propertyDescriptor.getName());
                continue;
            }

            attr = convertOutputPropertyValue(propertyDescriptor, attr, scrubber);
            propertyValues.put(propertyName, attr);
        }
        return propertyValues;
    }

    private static Object convertOutputPropertyValue(final PropertyDescriptor propertyDescriptor, Object propertyValue, SensitiveValueScrubber scrubber) {
        Variable variable = Optional.<Variable>ofNullable(createVariableByPropertyDescriptor(propertyDescriptor).orNull(null)).orElse(new StringVariable());
        try {
            variable.setUntypedValue(propertyValue);
	        Object value = variable.getValue();
	        if (value instanceof CharSequence && !propertyDescriptor.isPassword()) {
		        value = scrubber.scrubValues(value.toString());
	        }
	        return value;
        } catch (IllegalArgumentException e) {
            throw new IllegalArgumentException(format("Cannot store value '%s' of type %s into %s property '%s'",
                    propertyValue, propertyValue.getClass().getSimpleName(), propertyDescriptor.getKind(), propertyDescriptor.getName()));
        }
    }

    public static Option<String> getAttachmentIdFromExecutionLog(Writer executionLog) {
        //close writer to generate artifactId and store artifact. See, com.xebialabs.xlrelease.script.DefaultScriptService.ScriptTaskOutputWriter.saveArtifact
        closeWriter(executionLog);

        ThreadLocalWriterDecorator writerDecorator = (ThreadLocalWriterDecorator) executionLog;
        Optional<String> res = Optional.ofNullable((ExecutionOutputWriter) writerDecorator.getWriter())
                .map(executionOutputWriter -> (DefaultScriptService.ScriptTaskOutputWriter) executionOutputWriter.getWriter())
                .map(DefaultScriptService.ScriptTaskOutputWriter::getAttachmentId);
        return scala.jdk.javaapi.OptionConverters.toScala(res);
    }

    public static void addExceptionToExecutionLog(Exception exception, Writer executionLog, String logMessage, String taskId) {
        if (!(exception instanceof ScriptException) || !isKeyboardInterrupt((ScriptException) exception)) {
            logger.warn(logMessage, taskId, exception);
        }
        String defaultMessage = Objects.toString(sanitizeServerPath(exception.toString()), "");
        try {
            executionLog.append("Exception during execution:\n");
            executionLog.append(defaultMessage);
            executionLog.flush();
            closeWriter(executionLog);
        } catch (Exception e) {
            logger.warn("Unable to update Digital.ai Release task: '{}'", taskId, e);
        }
    }

    protected static String sanitizeServerPath(String message) {
        return message == null ? null : message.replace(System.getProperty("user.dir"), "{ServerWorkingDirectory}");
    }

    public static boolean isSystemExit0(ScriptException exception) {
        if (!(exception.getCause() instanceof PyException)) {
            return false;
        }

        PyException pyE = (PyException) exception.getCause();
        String pyEtoString = pyE.toString();
        PyType pyET = (PyType) pyE.type;
        String pyETName = pyET.getName();
        return pyETName.equals("SystemExit") && pyEtoString.contains("SystemExit: 0");
    }

    public static boolean isKeyboardInterrupt(ScriptException exception) {
        if (!(exception.getCause() instanceof PyException)) {
            return false;
        }

        PyException pyE = (PyException) exception.getCause();
        PyType pyET = (PyType) pyE.type;
        String pyETName = pyET.getName();
        return pyETName.equals("KeyboardInterrupt");
    }

    public static void closeWriter(Writer executionLog) {
        try {
            executionLog.close();
        } catch (IOException e) {
            logger.warn("Error closing script output logger", e);
        }
    }
}
