package com.xebialabs.xlrelease.service;

import java.util.Date;
import java.util.List;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;
import com.codahale.metrics.annotation.Timed;

import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.xlrelease.domain.Comment;
import com.xebialabs.xlrelease.domain.Task;
import com.xebialabs.xlrelease.domain.events.CommentCreatedEvent;
import com.xebialabs.xlrelease.domain.events.CommentDeletedEvent;
import com.xebialabs.xlrelease.domain.events.CommentUpdatedEvent;
import com.xebialabs.xlrelease.events.XLReleaseEventBus;
import com.xebialabs.xlrelease.repository.CommentRepository;
import com.xebialabs.xlrelease.repository.TaskRepository;
import com.xebialabs.xlrelease.serialization.json.repository.ResolveOptions;
import com.xebialabs.xlrelease.user.User;

import static com.xebialabs.deployit.booter.local.utils.Strings.isBlank;
import static com.xebialabs.xlrelease.builder.CommentBuilder.newComment;
import static com.xebialabs.xlrelease.domain.utils.CommentServiceHelper.appendCommentText;
import static com.xebialabs.xlrelease.domain.utils.CommentServiceHelper.updateCommentText;
import static com.xebialabs.xlrelease.repository.CiCloneHelper.cloneCi;
import static scala.jdk.javaapi.CollectionConverters.asJava;

@Repository
public class CommentService {
    public static final String COMMENT_PREFIX_SCRIPT = String.format("This comment was too long and has been cut off at the end. See the `script_output.log` attachment for the full log.%n%n---%n%n");
    public static final String COMMENT_PREFIX_MANUAL = String.format("This comment was too long and has been cut off at the end.%n%n---%n%n");
    public static final String COMMENT_SUFFIX = String.format("%n%n---%n%n[...truncated...]");

    private static final Logger logger = LoggerFactory.getLogger(CommentService.class);

    private final CiIdService ciIdService;

    private final XLReleaseEventBus eventBus;

    private final CommentRepository commentRepository;

    private final TaskRepository taskRepository;

    @Autowired
    public CommentService(CiIdService ciIdService, XLReleaseEventBus eventBus, CommentRepository commentRepository, TaskRepository taskRepository) {
        this.ciIdService = ciIdService;
        this.eventBus = eventBus;
        this.commentRepository = commentRepository;
        this.taskRepository = taskRepository;
    }


    @Timed
    public Comment create(Task task, Comment comment) {
        return createComment(task, comment.getText(), comment.getAuthor(), false, COMMENT_PREFIX_MANUAL);
    }

    @Timed
    public Comment create(String taskId, Comment comment) {
        Task task = taskRepository.findById(taskId, ResolveOptions.WITH_DECORATORS());
        return create(task, comment);
    }

    @Timed
    public Comment create(String taskId, String text, User author, boolean notify) {
        Task task = taskRepository.findById(taskId, ResolveOptions.WITH_DECORATORS());
        return create(task, text, author, notify);
    }

    @Timed
    public Comment create(Task task, String text, User author, boolean notify) {
        return createComment(task, text, author, notify, COMMENT_PREFIX_MANUAL);
    }

    @Timed
    public Comment update(Task task, String commentId, String text) {
        Comment comment = commentRepository.findById(commentId);
        Comment original = cloneCi(comment);

        int maxCommentSize = task.getMaxCommentSize();

        if (!text.equals(comment.getText())) {
            updateCommentText(comment, text, maxCommentSize, COMMENT_PREFIX_MANUAL, COMMENT_SUFFIX);
            task.updateComment(original, comment);
            commentRepository.update(comment.getId(), comment.getText());

            eventBus.publish(new CommentUpdatedEvent(task, original, comment));
        }

        return comment;
    }

    @Timed
    public String appendComment(Task task, String commentId, String commentText) {
        String updatedCommentId = commentId;
        try {
            int maxCommentSize = task.getMaxCommentSize();
            if (isBlank(updatedCommentId)) {
                Comment comment = createComment(task, commentText, User.LOG_OUTPUT, false, COMMENT_PREFIX_SCRIPT);
                updatedCommentId = comment.getId();
            } else {
                Comment comment = commentRepository.findById(updatedCommentId);
                Comment original = cloneCi(comment);
                appendCommentText(comment, commentText, maxCommentSize, COMMENT_PREFIX_SCRIPT, COMMENT_SUFFIX);
                task.updateComment(original, comment);
                commentRepository.update(comment.getId(), comment.getText());
                eventBus.publish(new CommentUpdatedEvent(task, original, comment));
            }
        } catch (Exception exception) {
            logger.warn("Unable to update Digital.ai Release task: '{}'", task.getId(), exception);
        }
        return updatedCommentId;
    }


    private Comment createComment(Task task, String commentText, User author, boolean notify, String prefix) {
        String commentAuthor = (author == User.SYSTEM) ? null : author.getName();
        return createComment(task, commentText, commentAuthor, notify, prefix);
    }

    private Comment createComment(Task task, String commentText, String commentAuthor, boolean notify, String prefix) {
        Comment comment = buildComment(task, commentText, commentAuthor, new Date(), prefix, CommentService.COMMENT_SUFFIX);
        task.getComments().add(comment);
        logger.debug("Creating comment: {}", comment);
        commentRepository.create(comment);
        eventBus.publish(new CommentCreatedEvent(task, comment, commentAuthor, notify));
        return comment;
    }

    public Comment buildComment(Task task, String commentText, String author, Date creationDate, String prefix, String suffix) {
        int maxCommentSize = task.getMaxCommentSize();
        Comment comment = newComment()
                .withId(ciIdService.getUniqueId(Type.valueOf(Comment.class), task.getId()))
                .withCreationDate(creationDate)
                .withLimit(maxCommentSize)
                .withAuthor(author)
                .build();
        appendCommentText(comment, commentText, maxCommentSize, prefix, suffix);
        return comment;
    }

    public void decorate(Task task) {
        commentRepository.decorate(task);
    }

    public List<Comment> findByTask(String taskId) {
        return asJava(commentRepository.findByTask(taskId));
    }

    @Timed
    public void delete(Task task, final String commentId) {
        Comment comment = commentRepository.findById(commentId);
        commentRepository.delete(comment);
        task.getComments().remove(comment);
        eventBus.publish(new CommentDeletedEvent(task, comment));
    }
}
