package com.xebialabs.xlrelease.service

import com.codahale.metrics.annotation.Timed
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry.getSubtypes
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlrelease.configuration.ThemeColors.{colorCode, themeColorFromCode}
import com.xebialabs.xlrelease.configuration._
import com.xebialabs.xlrelease.domain.BaseConfiguration
import com.xebialabs.xlrelease.domain.events.{ConfigurationCreatedEvent, ConfigurationDeletedEvent, ConfigurationUpdatedEvent}
import com.xebialabs.xlrelease.events.XLReleaseEventBus
import com.xebialabs.xlrelease.repository.Ids.isNullId
import com.xebialabs.xlrelease.repository._
import com.xebialabs.xlrelease.utils.PasswordVerificationUtils.replacePasswordPropertiesInCiIfNeeded
import grizzled.slf4j.Logging
import org.springframework.stereotype.Service

import java.util.{List => JList}
import scala.jdk.CollectionConverters._
import scala.util.Try

@Service
class ConfigurationService(configurationRepository: ConfigurationRepository, eventBus: XLReleaseEventBus) extends Logging {

  @Timed
  def createOrUpdate(id: String, configurationItem: BaseConfiguration): BaseConfiguration = {
    // TODO move validations to respective validators for types: REL-9705
    configurationItem match {
      case reportsSettings: ReportsSettings => reportsSettings.validate()
      case archivingSettings: ArchivingSettings => archivingSettings.validate()
      case pollingSettings: PollingSettings => pollingSettings.validate()
      case themeSettings: ThemeSettings =>
        themeSettings.setHeaderAccentColor(themeColorFromCode(themeSettings.getHeaderAccentColor.toUpperCase).name())
      case auditReportSettings: AuditReportSettings => auditReportSettings.validate()
      case executionLogPurgeSettings: ExecutionLogPurgeSettings => executionLogPurgeSettings.validate()
      case _ =>
    }

    configurationItem.setId(id)
    createOrUpdate(configurationItem)
    read(id)
  }

  @Timed
  def createOrUpdate(configuration: BaseConfiguration): Unit = {
    if (!isNullId(configuration.getId) && configurationRepository.exists(configuration.getId)) {
      val original: BaseConfiguration = configurationRepository.read(configuration.getId)

      replacePasswordPropertiesInCiIfNeeded(Some(original), configuration)

      configurationRepository.update(configuration)
      eventBus.publish(ConfigurationUpdatedEvent(configuration))
    } else {
      configurationRepository.create(configuration)
      eventBus.publish(ConfigurationCreatedEvent(configuration))
    }
  }

  def read(id: String): BaseConfiguration = {
    val baseConfiguration: BaseConfiguration = configurationRepository.read(id)
    baseConfiguration match {
      case themeSettings: ThemeSettings =>
        convertColorNameToCode(themeSettings)
      case other => other
    }
  }

  @Timed
  def exists(id: String): Boolean = configurationRepository.exists(id)

  @Timed
  def delete(id: String): Unit = {
    val conf: BaseConfiguration = configurationRepository.read(id)
    configurationRepository.delete(id)
    eventBus.publish(ConfigurationDeletedEvent(conf))
  }

  @Timed
  def getThemeSettings: ThemeSettings = {
    val themeSettings = configurationRepository.getThemeSettings()
    convertColorNameToCode(themeSettings)
  }

  private def convertColorNameToCode(themeSettings: ThemeSettings): ThemeSettings = {
    themeSettings.setHeaderAccentColor(colorCode(ThemeColor.valueOf(themeSettings.getHeaderAccentColor.toUpperCase())))
    themeSettings
  }

  @Timed
  def saveCustomLogo(customLogoSettings: CustomLogoSettings): CustomLogoSettings = {
    customLogoSettings.validateContentType()
    createOrUpdate(CustomLogoSettings.CUSTOM_LOGO_SETTINGS_ID, customLogoSettings).asInstanceOf[CustomLogoSettings]
  }

  @Timed
  def getFeatureSettings(): JList[FeatureSettings] = {
    val features = configurationRepository.findAllByType[FeatureSettings](Type.valueOf(classOf[FeatureSettings])).asScala

    getSubtypes(Type.valueOf(classOf[FeatureSettings])).asScala.map { featureType =>
      features.find(_.getType == featureType).getOrElse {
        val feature = featureType.getDescriptor.newInstance[FeatureSettings]("")
        feature.generateId()
        feature
      }
    }.toList.sortBy(_.getType.getDescriptor.getLabel).asJava
  }

  @Timed
  def getFeatureSettings(featureType: String): FeatureSettings = {
    def getDefaultSettings: FeatureSettings = {
      val feature = Type.valueOf(featureType).getDescriptor.newInstance[FeatureSettings]("")
      feature.generateId()
      feature
    }

    Try(configurationRepository.findFirstByType(Type.valueOf(featureType)).orElse(getDefaultSettings)).getOrElse(getDefaultSettings)
  }

  @Timed
  def saveFeatureSettings(features: JList[FeatureSettings]): Unit = {
    features.asScala.foreach(createOrUpdate)
  }

}
