package com.xebialabs.xlrelease.service;

import java.util.*;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;
import com.codahale.metrics.annotation.Timed;

import com.xebialabs.deployit.exception.NotFoundException;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.xlrelease.domain.Link;
import com.xebialabs.xlrelease.domain.ParallelGroup;
import com.xebialabs.xlrelease.domain.Task;
import com.xebialabs.xlrelease.domain.events.LinkCreatedEvent;
import com.xebialabs.xlrelease.domain.events.LinkDeletedEvent;
import com.xebialabs.xlrelease.domain.events.TaskUpdatedEvent;
import com.xebialabs.xlrelease.domain.events.XLReleaseEvent;
import com.xebialabs.xlrelease.events.XLReleaseEventBus;
import com.xebialabs.xlrelease.repository.LinkRepository;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkState;
import static com.xebialabs.xlrelease.builder.LinkBuilder.newLink;
import static com.xebialabs.xlrelease.repository.CiCloneHelper.cloneCi;
import static java.lang.String.format;

@Repository
public class LinkService {

    private static final Logger logger = LoggerFactory.getLogger(LinkService.class);

    private CiIdService ciIdService;
    private TaskService taskService;
    private XLReleaseEventBus eventBus;
    private LinkRepository linkRepository;

    @Autowired
    public LinkService(CiIdService ciIdService,
                       TaskService taskService,
                       XLReleaseEventBus eventBus,
                       LinkRepository linkRepository) {
        this.ciIdService = ciIdService;
        this.taskService = taskService;
        this.eventBus = eventBus;
        this.linkRepository = linkRepository;
    }

    @Timed
    public Link create(String containerId, String sourceId, String targetId) {
        Task container = taskService.getTaskWithoutDecoration(containerId);
        Task source = taskService.getTaskWithoutDecoration(sourceId);
        Task target = taskService.getTaskWithoutDecoration(targetId);

        Link link = newLink()
                .withId(ciIdService.getUniqueId(Type.valueOf(Link.class), containerId))
                .withSource(source)
                .withTarget(target)
                .build();

        checkArgument(container instanceof ParallelGroup, "You can't add a link to a container that's not a parallel group.");
        checkState(!container.hasBeenStarted(), "You can't add a link to a container that has already been started.");
        checkState(uniqueLinkWithinContainer(container, link), "You can't add the same link twice within a parallel group.");
        checkArgument(source.getContainer().equals(container), "You can't add a link whose source isn't a direct child of the container.");
        checkArgument(target.getContainer().equals(container), "You can't add a link whose target isn't a direct child of the container.");

        ParallelGroup parallelGroup = (ParallelGroup) container;
        LockedTaskOperationChecks.checkCreateTaskLink(parallelGroup);
        parallelGroup.getLinks().add(link);
        link.setParallelGroup(parallelGroup);

        linkRepository.create(link);

        eventBus.publish(new LinkCreatedEvent(link));

        return link;
    }

    @Timed
    public void delete(String linkId) {
        try {
            Link link = linkRepository.findById(linkId);

            ParallelGroup parallelGroup = link.getParallelGroup();
            checkState(!parallelGroup.hasBeenStarted(), "Can't remove a link from a container that has already been started");
            LockedTaskOperationChecks.checkRemoveTaskLink(parallelGroup);

            Task target = link.getTarget();

            List<XLReleaseEvent> events = new ArrayList<>();
            events.add(new LinkDeletedEvent(link));

            boolean isTargetNeedUpdate = !target.isWaitForScheduledStartDate();
            if (isTargetNeedUpdate) {
                Task original = cloneCi(target);
                target.setWaitForScheduledStartDate(true);
                events.add(new TaskUpdatedEvent(original, target));
            }
            Set<Link> links = link.getParallelGroup().getLinks();
            // CI Ids are rewritten after deserialization, which means that hashes of stored CIs are different
            // so remove on Set is not working and we have to replace the whole set
            Set<Link> newLinks = links.stream().filter(l -> !l.getId().equals(link.getId())).collect(Collectors.toSet());
            link.getParallelGroup().setLinks(newLinks);
            linkRepository.delete(link);

            events.forEach(eventBus::publish);
        } catch (NotFoundException e) {
            logger.debug(format("Link with id %s not found in repository, skipping delete", linkId));
        }

    }

    private boolean uniqueLinkWithinContainer(Task container, Link newLink) {
        Collection<Link> links = ((ParallelGroup) container).getLinks();

        for (Link link : links) {
            if (Objects.equals(newLink.getSource(), link.getSource()) && Objects.equals(newLink.getTarget(), link.getTarget())) {
                return false;
            }
        }

        return true;
    }
}
