package com.xebialabs.xlrelease.service

import com.codahale.metrics.annotation.Timed
import com.xebialabs.xlplatform.ui.{ExtensionLibrary, Settings, UiExtensions}
import com.xebialabs.xlplatform.utils.ResourceManagement.using
import com.xebialabs.xlrelease.Environment
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.utils.ResourceUtils
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.core.io.{Resource, ResourceLoader}
import org.springframework.stereotype.Service

import java.io.BufferedReader
import java.net.URI
import java.nio.charset.StandardCharsets.UTF_8
import java.nio.file._
import scala.io.Source
import scala.jdk.CollectionConverters._
import scala.util.{Failure, Success, Try, Using}

@Service class  UiExtensionsService @Autowired()(val xlrConfig: XlrConfig, resourceLoader: ResourceLoader) extends Logging {

  private val MAX_DEPTH: Int = 5
  private val fsCreatorLock = new Object

  @Timed
  def getAllJs: String = {
    processPaths(".js", Environment.DEV_PLUGINS_CONVERTED_JS.toPath)
  }

  @Timed
  def getAllCss: String = {
    processPaths(".css", Environment.DEV_PLUGINS_CONVERTED_CSS.toPath)
  }

  private def processPaths(extension: String, devPath: Path): String = {
    val resourceData = concatFiles(resourcesList(extension))
    if (Environment.isDevelopment) {
      val stringBuilder = new StringBuilder(resourceData)
      stringBuilder.append("\n")
      if (Files.exists(devPath)) {
        using(Files.newBufferedReader(devPath, UTF_8)) { r => flushToBuilder(r, stringBuilder) }
      } else {
        logger.warn(s"Unable to find $devPath")
      }
      stringBuilder.toString()
    } else {
      resourceData
    }
  }

  @Timed
  def getXlReleaseModuleJs: String = {
    val templateResource: Resource = ResourceUtils.webResource("xlrelease-module.js.template")
    getXlReleaseModuleJs(templateResource)
  }

  private[service] def getXlReleaseModuleJs(templateResource: Resource): String = {
    using(templateResource.getInputStream) { is =>
      val template = Source.fromInputStream(is, UTF_8.name()).mkString
      val PLACEHOLDER = "/*PLUGIN_ANGULAR_DEPENDENCIES*/"
      val dependencies = findUiMenuLibraries()
      val replacement = if (dependencies.isEmpty) "" else dependencies.mkString("'", "', '", "'")
      template.replace(PLACEHOLDER, replacement)
    }
  }

  private def findUiMenuLibraries(): List[String] = {
    val filePattern = Settings.XlDeploy.UiExtension.file(xlrConfig.rootConfig)
    UiExtensions(filePattern).flatMap {
      case ExtensionLibrary(libraryName) => Option(libraryName)
      case _ => None
    }
  }

  private def concatFiles(paths: Seq[Path]): String = {
    val stringBuilder = new StringBuilder
    paths.foreach { p =>
      using(Files.newBufferedReader(p, UTF_8)) { r => flushToBuilder(r, stringBuilder) }
    }
    stringBuilder.toString()
  }

  private def flushToBuilder(reader: BufferedReader, stringBuilder: StringBuilder): Unit = {
    var aux = reader.readLine()
    while (aux != null) {
      stringBuilder.append(aux + "\n")
      aux = reader.readLine()
    }
  }

  private def resourcesList(fileExtension: String): List[Path] = {
    import scala.jdk.StreamConverters._
    getRoot.flatMap(r =>
      Using.resource(Files.walk(r, MAX_DEPTH)) { files =>
        files
          .filter((p: Path) =>
            p.toString.toLowerCase.endsWith(fileExtension.toLowerCase)
              && !Files.isDirectory(p)
              && !p.toString.toLowerCase.contains("hot-update")
          )
          .sorted((o1: Path, o2: Path) =>
            o1.getFileName.compareTo(o2.getFileName)
          )
          .toScala(List)
      }

    )
  }

  private def getRoot: List[Path] = {
    val classLoader = resourceLoader.getClassLoader
    val resources = classLoader.getResources(Environment.PLUGIN_WEB_FOLDER).asScala ++
      classLoader.getResources(Environment.PRD_PLUGINS_CONVERTED_JS).asScala ++
      classLoader.getResources(Environment.PRD_PLUGINS_CONVERTED_CSS).asScala
    resources.map(path => path.toURI.ensureFileSystemAvailable.getNioPath).filter(Files.exists(_)).toList
  }

  implicit class UriHelper(uri: URI) {
    def ensureFileSystemAvailable: URI = fsCreatorLock.synchronized({
      def recreateFileSystemAccess =
        (uri: URI, logMsg: String, origExc: Exception) => Try(FileSystems.newFileSystem(uri, Map.empty[String, String].asJava)) match {
          case Success(_) => logger.debug(s"Successfully recreated filesystem for $uri after initial failure - original message: $logMsg")
          case Failure(e) =>
            logger.warn(s"Could not recreate filesystem for $uri", e)
            logger.warn(s"Original failure: $logMsg", origExc)
        }

      if (uri.getScheme != "file") Try(FileSystems.getFileSystem(uri)) match {
        case Failure(e: FileSystemNotFoundException) =>
          recreateFileSystemAccess(uri, s"Retrying to get filesystem for $uri because: ${e.getMessage}", e)
        case Failure(e) =>
          logger.warn(s"Cannot get filesystem for $uri because: ${e.getMessage}", e)
        case Success(fs) if !fs.isOpen =>
          recreateFileSystemAccess(uri, s"FileSystem ${fs.toString} for $uri is not open - retrying", null)
        case Success(fs) =>
          logger.debug(s"Found filesystem $fs for $uri")
      }
      uri
    })

    def getNioPath: Path = Paths.get(uri)
  }

}

