package com.xebialabs.xlrelease.views;

import java.util.*;

import com.xebialabs.xlrelease.domain.status.FlagStatus;
import com.xebialabs.xlrelease.domain.status.TaskStatus;
import com.xebialabs.xlrelease.domain.variables.Variable;

import static com.google.common.collect.Maps.newHashMap;

public class TaskViewBuilder {
    private String id;
    private String title;
    private String description;
    private String type;
    private TaskStatus status;
    private FlagStatus flagStatus = FlagStatus.OK;
    private String flagComment;
    private Date dueDate;
    private Date scheduledStartDate;
    private boolean delayDuringBlackout;
    private String owner;
    private String team;
    private String environment;
    private String deploymentPackage;
    private String server;
    private Set<String> addresses = new HashSet<>();
    private String subject;
    private String body;
    private Map<String, Object> inputProperties = newHashMap();
    private Map<String, Object> outputProperties = newHashMap();
    private String scriptDefinitionType;
    private String ownerFullName;
    private Long duration;
    private List<Variable> variablesList = new ArrayList<>();
    private String templateId;
    private String folderId;
    private String newReleaseTitle;
    private String createdReleaseId;
    private boolean startRelease;
    private Map<String, String> variableMapping = newHashMap();
    private List<String> tags = new ArrayList<>();

    private TaskViewBuilder() {
    }

    public static TaskViewBuilder newTaskView() {
        return new TaskViewBuilder();
    }

    public TaskViewBuilder withId(String id) {
        this.id = id;
        return this;
    }

    public TaskViewBuilder withTitle(String title) {
        this.title = title;
        return this;
    }

    public TaskViewBuilder withDescription(String description) {
        this.description = description;
        return this;
    }

    public TaskViewBuilder withType(String type) {
        this.type = type;
        return this;
    }

    public TaskViewBuilder withStatus(TaskStatus status) {
        this.status = status;
        return this;
    }

    public TaskViewBuilder withFlagStatus(FlagStatus flagStatus) {
        this.flagStatus = flagStatus;
        return this;
    }

    public TaskViewBuilder withFlagComment(String flagComment) {
        this.flagComment = flagComment;
        return this;
    }

    public TaskViewBuilder withDueDate(Date dueDate) {
        this.dueDate = dueDate;
        return this;
    }

    public TaskViewBuilder withScheduledStartDate(Date scheduledStartDate) {
        this.scheduledStartDate = scheduledStartDate;
        return this;
    }

    public TaskViewBuilder withDelayDuringBlackout(boolean delayDuringBlackout) {
        this.delayDuringBlackout = delayDuringBlackout;
        return this;
    }

    public TaskViewBuilder assignedTo(String owner) {
        this.owner = owner;
        this.ownerFullName = null;
        return this;
    }

    public TaskViewBuilder withTeam(String team) {
        this.team = team;
        return this;
    }

    public TaskViewBuilder withEnvironment(String environment) {
        this.environment = environment;
        return this;
    }

    public TaskViewBuilder withServer(String server) {
        this.server = server;
        return this;
    }

    public TaskViewBuilder withDeploymentPackage(String deploymentPackage) {
        this.deploymentPackage = deploymentPackage;
        return this;
    }

    public TaskViewBuilder withAddresses(Set<String> addresses) {
        this.addresses = addresses;
        return this;
    }

    public TaskViewBuilder withSubject(String subject) {
        this.subject = subject;
        return this;
    }

    public TaskViewBuilder withBody(String body) {
        this.body = body;
        return this;
    }

    public TaskViewBuilder withScriptDefinitionType(String scriptDefinitionType) {
        this.scriptDefinitionType = scriptDefinitionType;
        return this;
    }

    public TaskViewBuilder withInputProperties(Map<String, Object> inputProperties) {
        this.inputProperties = newHashMap(inputProperties);
        return this;
    }

    public TaskViewBuilder withOutputProperties(Map<String, Object> outputProperties) {
        this.outputProperties = newHashMap(outputProperties);
        return this;
    }

    public TaskViewBuilder withPlannedDuration(Long duration) {
        this.duration = duration;
        return this;
    }

    public TaskViewBuilder withVariables(List<Variable> variables) {
        this.variablesList = variables;
        return this;
    }

    public TaskViewBuilder withVariableMapping(Map<String, String> variableMapping) {
        this.variableMapping = variableMapping;
        return this;
    }

    public TaskViewBuilder withNewReleaseTitle(String newReleaseTitle) {
        this.newReleaseTitle = newReleaseTitle;
        return this;
    }

    public TaskViewBuilder withTemplateId(String templateId) {
        this.templateId = templateId;
        return this;
    }

    public TaskViewBuilder withFolderId(final String folderId) {
        this.folderId = folderId;
        return this;
    }

    public TaskViewBuilder withCreatedReleaseId(String createdReleaseId) {
        this.createdReleaseId = createdReleaseId;
        return this;
    }

    public TaskViewBuilder withStartRelease(boolean startRelease) {
        this.startRelease = startRelease;
        return this;
    }

    public TaskViewBuilder withTags(List<String> tags) {
        this.tags = tags;
        return this;
    }

    public TaskFullView build() {
        TaskFullView taskView;

        taskView = new TaskFullView();
        taskView.setId(id);
        taskView.setTitle(title);
        taskView.setDescription(description);
        taskView.setType(type);
        taskView.setStatus(status);
        taskView.setFlag(new Flag(flagStatus, flagComment));
        taskView.setDueDate(dueDate);
        taskView.setScheduledStartDate(scheduledStartDate);
        taskView.setDelayDuringBlackout(delayDuringBlackout);
        if (null != owner) {
            taskView.setOwner(new UserView(owner, ownerFullName));
        }
        taskView.setPassword(new VariableOrPasswordView());
        taskView.setTeam(team);
        taskView.setEnvironment(environment);
        taskView.setServer(server);
        taskView.setDeploymentPackage(deploymentPackage);
        taskView.setAddresses(addresses);
        taskView.setSubject(subject);
        taskView.setBody(body);
        taskView.setScriptDefinitionType(scriptDefinitionType);
        taskView.setInputProperties(inputProperties);
        taskView.setOutputProperties(outputProperties);
        taskView.setPlannedDuration(duration);
        taskView.setVariables(variablesList);
        taskView.setVariableMapping(variableMapping);
        taskView.setNewReleaseTitle(newReleaseTitle);
        taskView.setTemplateId(templateId);
        taskView.setFolderId(folderId);
        taskView.setCreatedReleaseId(createdReleaseId);
        taskView.setStartRelease(startRelease);
        taskView.setTags(tags);
        return taskView;
    }
}
