package com.xebialabs.xlrelease.api.v1.impl;

import java.util.List;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import com.codahale.metrics.annotation.Timed;
import com.google.common.collect.Lists;

import com.xebialabs.deployit.plugin.api.reflect.Descriptor;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.xlrelease.api.v1.FacetApi;
import com.xebialabs.xlrelease.api.v1.forms.FacetFilters;
import com.xebialabs.xlrelease.domain.facet.ConfigurationFacet;
import com.xebialabs.xlrelease.domain.facet.Facet;
import com.xebialabs.xlrelease.domain.facet.TaskReportingRecord;
import com.xebialabs.xlrelease.security.FacetPermissionChecker;
import com.xebialabs.xlrelease.security.PermissionChecker;
import com.xebialabs.xlrelease.service.FacetService;

import static com.xebialabs.xlrelease.repository.Ids.releaseIdFrom;

@Controller
public class FacetApiImpl implements FacetApi {

    private final FacetService facetService;
    private final PermissionChecker permissions;
    private final FacetPermissionChecker facetPermissions;

    @Autowired
    public FacetApiImpl(FacetService facetService, PermissionChecker permissions, FacetPermissionChecker facetPermissions) {
        this.facetService = facetService;
        this.permissions = permissions;
        this.facetPermissions = facetPermissions;
    }

    @Timed
    @Override
    public Facet getFacet(String facetId) {
        Facet facet = facetService.get(facetId);
        facetPermissions.checkThatScriptTaskBelongsToReleaseOrOtherwise(releaseIdFrom(facet.getTargetId()), () ->
                permissions.checkView(releaseIdFrom(facet.getTargetId())));
        checkIsConfigurationFacet(facet, "Cannot fetch facet of this type (" + facet.getType() + ")");

        return facet;
    }

    @Timed
    @Override
    public Facet createFacet(ConfigurationFacet configurationFacet) {
        performConfigurationFacetPermissionChecks(configurationFacet);
        return facetService.create(configurationFacet);
    }

    @Timed
    @Override
    public Facet newFacet(String type) {
        Facet facet = Type.valueOf(type).getDescriptor().newInstance("");
        checkIsConfigurationFacet(facet, "You can only create Configuration type facets.");
        return facet;
    }

    @Timed
    @Override
    public Facet updateFacet(String facetId, ConfigurationFacet configurationFacet) {
        performConfigurationFacetPermissionChecks(configurationFacet);
        configurationFacet.setId(facetId);
        return facetService.update(configurationFacet);
    }

    @Timed
    @Override
    public void deleteFacet(String facetId) {
        Facet facet = facetService.get(facetId);
        performConfigurationFacetPermissionChecks(facet);
        checkIsConfigurationFacet(facet, "You can only remove Configuration facets");

        facetService.delete(facetId);
    }

    @Timed
    @Override
    public List<Facet> searchFacets(FacetFilters facetFilters) {
        facetFilters.validate();
        String containerId = facetFilters.getParentId() != null ? facetFilters.getParentId() : facetFilters.getTargetId();
        facetPermissions.checkThatScriptTaskBelongsToReleaseOrOtherwise(releaseIdFrom(containerId), () ->
                permissions.checkView(releaseIdFrom(containerId)));
        checkFiltersOnlyContainSearchableTypes(facetFilters);

        return facetService.search(facetFilters);
    }

    @Timed
    @Override
    public List<Descriptor> getFacetTypes(String baseType) {
        return facetService.getFacetTypes(baseType);
    }

    private void performConfigurationFacetPermissionChecks(Facet facet) {
        String releaseId = releaseIdFrom(facet.getTargetId());
        facetPermissions.checkThatScriptTaskBelongsToReleaseOrOtherwise(releaseId,
                () -> permissions.checkEditTaskConfigurationFacet(releaseId));
    }

    private void checkIsConfigurationFacet(final Facet facet, final String exceptionMsg) {
        if (!facet.getType().instanceOf(Type.valueOf(ConfigurationFacet.class))) {
            throw new IllegalArgumentException(exceptionMsg);
        }
    }

    private void checkFiltersOnlyContainSearchableTypes(FacetFilters facetFilters) {
        if (facetFilters.getTypes() == null || facetFilters.getTypes().isEmpty()) {
            facetFilters.setTypes(Lists.newArrayList(Type.valueOf(ConfigurationFacet.class)));
        } else {
            if (facetFilters.getTypes().stream().anyMatch(facetType -> facetType.instanceOf(Type.valueOf(TaskReportingRecord.class)))) {
                throw new IllegalArgumentException("Facet filters contain type not searchable by this API");
            }
        }
    }

}
