package com.xebialabs.xlrelease.api.v1.impl;


import java.util.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.cache.CacheManager;
import org.springframework.stereotype.Controller;
import com.codahale.metrics.annotation.Timed;

import com.xebialabs.deployit.checks.Checks;
import com.xebialabs.deployit.core.api.dto.RolePermissions;
import com.xebialabs.deployit.core.rest.api.SecurityResource;
import com.xebialabs.deployit.engine.api.dto.Paging;
import com.xebialabs.deployit.engine.api.security.RolePrincipals;
import com.xebialabs.deployit.exception.NotFoundException;
import com.xebialabs.deployit.repository.ItemAlreadyExistsException;
import com.xebialabs.deployit.security.Role;
import com.xebialabs.deployit.security.RoleService;
import com.xebialabs.deployit.security.permission.Permission;
import com.xebialabs.deployit.security.permission.PlatformPermissions;
import com.xebialabs.xlrelease.api.v1.RolesApi;
import com.xebialabs.xlrelease.api.v1.views.PrincipalView;
import com.xebialabs.xlrelease.api.v1.views.RoleView;
import com.xebialabs.xlrelease.domain.events.GlobalRolesOrPermissionsUpdatedEvent;
import com.xebialabs.xlrelease.events.XLReleaseEventBus;
import com.xebialabs.xlrelease.exception.LogFriendlyNotFoundException;
import com.xebialabs.xlrelease.security.PermissionChecker;
import com.xebialabs.xlrelease.service.TeamService;
import com.xebialabs.xlrelease.service.UserInfoResolver;

import static com.xebialabs.deployit.checks.Checks.checkArgument;
import static com.xebialabs.xlrelease.security.PermissionChecker.GLOBAL_SECURITY_ALIAS;
import static com.xebialabs.xlrelease.utils.Collectors.toMap;
import static java.util.Collections.emptyList;
import static java.util.Collections.singletonList;
import static java.util.Comparator.comparing;
import static java.util.function.Function.identity;
import static java.util.stream.Collectors.joining;
import static java.util.stream.Collectors.toList;

@Controller
public class RolesApiImpl implements RolesApi {
    private static final Logger logger = LoggerFactory.getLogger(RolesApiImpl.class);

    private PermissionChecker permissions;
    private RoleService roleService;
    private SecurityResource securityResource;
    private UserInfoResolver userInfoResolver;
    private TeamService teamService;
    private XLReleaseEventBus xlReleaseEventBus;
    private Optional<CacheManager> cacheManager;

    public RolesApiImpl(PermissionChecker permissions,
                        RoleService roleService,
                        SecurityResource securityResource,
                        UserInfoResolver userInfoResolver,
                        TeamService teamService,
                        XLReleaseEventBus xlReleaseEventBus,
                        @Qualifier("securityCacheManager") Optional<CacheManager> cacheManager) {
        this.permissions = permissions;
        this.roleService = roleService;
        this.securityResource = securityResource;
        this.userInfoResolver = userInfoResolver;
        this.teamService = teamService;
        this.xlReleaseEventBus = xlReleaseEventBus;
        this.cacheManager = cacheManager;
    }

    @Timed
    @Override
    public List<RoleView> getRoles(Integer page, Integer resultsPerPage) {
        logger.trace("Entering getRoles(page={}, resultsPerPage={})", page, resultsPerPage);
        invalidateCachesIfNecessary();
        Paging paging = new Paging(page + 1, resultsPerPage);
        List<Role> roles = roleService.getRoles(null, paging, null);
        roles.sort(comparing(Role::getName));
        Map<String, RolePermissions> rolePermissions = permissionsByRole(securityResource.readRolePermissions(GLOBAL_SECURITY_ALIAS(), null, null, null));
        Map<String, RolePrincipals> rolePrincipals = principalsByRole(securityResource.readRolePrincipals(null, null, null));

        List<RoleView> result = roles.stream()
                .map(r -> assembleRoleView(r, rolePermissions, rolePrincipals))
                .collect(toList());
        logger.trace("Exiting getRoles(page={}, resultsPerPage={}). Found {} roles. Results:\n{}", page, resultsPerPage, result.size(), result);
        return result;
    }

    @Timed
    @Override
    public RoleView getRole(String roleName) {
        logger.trace("Entering getRole(roleName={})", roleName);
        invalidateCachesIfNecessary();
        List<Role> globalRoles = roleService.getRoles(roleName, null, null);
        Optional<Role> matchingRole = globalRoles.stream().filter(role -> role.getName().equals(roleName)).findFirst();
        if (!matchingRole.isPresent()) {
            throw new NotFoundException("Could not find the role [%s]", roleName);
        }
        Map<String, RolePermissions> rolePermissions = permissionsByRole(securityResource.readRolePermissions(GLOBAL_SECURITY_ALIAS(), roleName, null, null));
        Map<String, RolePrincipals> rolePrincipals = principalsByRole(securityResource.readRolePrincipals(roleName, null, null));
        RoleView result =  assembleRoleView(matchingRole.get(), rolePermissions, rolePrincipals);
        logger.trace("Exiting getRole(roleName={}). Result:\n{}", roleName, result);
        return result;
    }

    @Timed
    @Override
    public void create(String roleName, RoleView roleView) {
        logger.trace("Entering create(roleName={}, role={})", roleName, roleView);
        invalidateCachesIfNecessary();
        Checks.checkTrue(roleName.equals(roleView.getName()),
                "Role name '%s' given in the path is not equal to the role '%s' defined in the provided object",
                roleName, roleView.getName());
        create(singletonList(roleView));
        logger.trace("Exiting create(roleName={}, role={})", roleName, roleView);
    }

    @Timed
    @Override
    public void create(List<RoleView> roleViews) {
        logger.trace("Entering create(roles={})", roleViews);
        invalidateCachesIfNecessary();
        permissions.check(PlatformPermissions.EDIT_SECURITY);
        List<String> roleNames = roleViews.stream().map(RoleView::getName).collect(toList());
        Optional<Role> role = roleService.getRoles().stream().filter(r -> roleNames.contains(r.getName())).findFirst();
        if (role.isPresent()) {
            throw new ItemAlreadyExistsException("Role '%s' already exist. Maybe you wanted to update it?", role.get().getName());
        }
        checkPermissions(roleViews);

        List<RolePrincipals> rolePrincipals = securityResource.readRolePrincipals(null, null, null);
        roleViews.forEach(roleView -> {
            String roleName = roleView.getName();
            List<String> principals = roleView.getPrincipals().stream().map(PrincipalView::getUsername).collect(toList());
            final RolePrincipals newRolePrincipal = newRolePrincipal(roleName, principals);
            rolePrincipals.add(newRolePrincipal);
        });

        rolePrincipals.forEach(rp -> teamService.generateIdIfNecessary(rp.getRole()));
        securityResource.writeRolePrincipals(rolePrincipals);

        Map<String, RolePrincipals> principalsByRole = principalsByRole(securityResource.readRolePrincipals(null, null, null));

        List<RolePermissions> rolePermissions = securityResource.readRolePermissions(GLOBAL_SECURITY_ALIAS(), null, null, null);
        roleViews.forEach(roleView -> {
            String roleName = roleView.getName();
            String roleId = principalsByRole.get(roleName).getRole().getId();
            rolePermissions.add(newRolePermission(roleId, roleName, roleView.getPermissions()));
        });
        securityResource.writeRolePermissions(GLOBAL_SECURITY_ALIAS(), rolePermissions);
        xlReleaseEventBus.publish(GlobalRolesOrPermissionsUpdatedEvent.apply());
        logger.trace("Exiting create(roles={})", roleViews);
    }

    @Timed
    @Override
    public void update(String roleName, RoleView roleView) {
        logger.trace("Entering update(roleName={}, roles={})", roleName, roleView);
        invalidateCachesIfNecessary();
        roleView.setName(roleName);
        update(singletonList(roleView));
        logger.trace("Exiting update(roleName={}, roles={})", roleName, roleView);
    }

    @Timed
    @Override
    public void update(List<RoleView> roleViews) {
        logger.trace("Entering update(roles={})", roleViews);
        invalidateCachesIfNecessary();
        permissions.check(PlatformPermissions.EDIT_SECURITY);
        checkPermissions(roleViews);

        List<String> roleNamesToUpdate = roleViews.stream().map(RoleView::getName).collect(toList());
        List<Role> roles = roleService.getRoles();
        Map<String, Role> rolesByName = rolesByName(roles);

        Optional<String> firstNonExistent = roleNamesToUpdate.stream().filter(r -> !rolesByName.containsKey(r)).findFirst();
        if (firstNonExistent.isPresent()) {
            throw new LogFriendlyNotFoundException("Role '%s' does not exist. Maybe you wanted to create it?", firstNonExistent.get());
        }

        //TODO since we can query backend with role pattern we might want to change this
        List<RolePrincipals> rolePrincipals = securityResource.readRolePrincipals(null, null, null);
        Map<String, RolePrincipals> principalsByRole = principalsByRole(rolePrincipals);
        roleViews.forEach(roleView -> {
            String roleName = roleView.getName();
            RolePrincipals rolePrincipal = principalsByRole.get(roleName);
            rolePrincipal.setPrincipals(roleView.getPrincipals().stream().map(PrincipalView::getUsername).collect(toList()));
        });
        rolePrincipals.forEach(rp -> teamService.generateIdIfNecessary(rp.getRole()));
        securityResource.writeRolePrincipals(rolePrincipals);

        List<RolePermissions> rolePermissions = securityResource.readRolePermissions(GLOBAL_SECURITY_ALIAS(), null, null, null);
        final Map<String, RolePermissions> permissionsByRole = permissionsByRole(rolePermissions);
        roleViews.forEach(roleView -> {
            String roleName = roleView.getName();
            RolePermissions rolePermission = permissionsByRole.get(roleName);
            if (null == rolePermission) {
                rolePermission = emptyRolePermissions(rolesByName.get(roleName));
                rolePermissions.add(rolePermission);
            }
            rolePermission.setPermissions(new ArrayList<>(roleView.getPermissions()));
        });
        securityResource.writeRolePermissions(GLOBAL_SECURITY_ALIAS(), rolePermissions);
        xlReleaseEventBus.publish(GlobalRolesOrPermissionsUpdatedEvent.apply());
        logger.trace("Exiting update(roles={})", roleViews);
    }

    @Timed
    @Override
    public void delete(String roleName) {
        logger.trace("Entering delete(roleName={})", roleName);
        invalidateCachesIfNecessary();
        permissions.check(PlatformPermissions.EDIT_SECURITY);
        getRole(roleName);
        List<RolePrincipals> rolePrincipals = securityResource.readRolePrincipals(roleName, null, null);
        rolePrincipals.removeIf(rolePrincipal -> roleName.equals(rolePrincipal.getRole().getName()));
        securityResource.writeRolePrincipals(rolePrincipals);
        xlReleaseEventBus.publish(GlobalRolesOrPermissionsUpdatedEvent.apply());
        logger.trace("Exiting delete(roleName={})", roleName);
    }

    @Timed
    @Override
    public void rename(String roleName, String newName) {
        logger.trace("Entering rename(roleName={}, newName={})", roleName, newName);
        invalidateCachesIfNecessary();
        permissions.check(PlatformPermissions.EDIT_SECURITY);
        Optional<Role> role = roleService.getRoles().stream().filter(r -> r.getName().equals(roleName)).findFirst();
        if (!role.isPresent()) {
            throw new LogFriendlyNotFoundException("Role '%s' does not exist. Maybe you wanted to create it first?", roleName);
        }
        Optional<Role> targetRole = roleService.getRoles().stream().filter(r -> r.getName().equals(newName)).findFirst();
        if (targetRole.isPresent()) {
            throw new ItemAlreadyExistsException("Role '%s' already exist.", newName);
        }

        List<RolePrincipals> rolePrincipals = securityResource.readRolePrincipals(roleName, null, null);
        RolePrincipals rolePrincipal = principalsByRole(rolePrincipals).get(roleName);
        rolePrincipal.getRole().setName(newName);
        securityResource.writeRolePrincipals(rolePrincipals);
        xlReleaseEventBus.publish(GlobalRolesOrPermissionsUpdatedEvent.apply());
        logger.trace("Exiting rename(roleName={}, newName={})", roleName, newName);
    }

    private RoleView assembleRoleView(final Role role, Map<String, RolePermissions> rolePermissions, Map<String, RolePrincipals> rolePrincipals) {
        final RoleView roleView = new RoleView();
        roleView.setId(role.getId());
        roleView.setName(role.getName());
        roleView.setPrincipals(rolePrincipals.get(role.getName()).getPrincipals().stream().map(this::assemblePrincipal).collect(toList()));
        List<String> permissions = rolePermissions.getOrDefault(role.getName(), emptyRolePermissions(role)).getPermissions();
        roleView.setPermissions(new HashSet<>(permissions));
        return roleView;
    }

    private RolePermissions emptyRolePermissions(Role role) {
        return newRolePermission(role.getId(), role.getName(), emptyList());
    }

    private RolePrincipals newRolePrincipal(String roleName, List<String> principals) {
        com.xebialabs.deployit.engine.api.security.Role role = new com.xebialabs.deployit.engine.api.security.Role();
        role.setName(roleName);
        return new RolePrincipals(role, principals);
    }

    private RolePermissions newRolePermission(String id, String roleName, Collection<String> permissions) {
        com.xebialabs.deployit.engine.api.security.Role role = new com.xebialabs.deployit.engine.api.security.Role(id, roleName);
        return new RolePermissions(role, new ArrayList<>(permissions));
    }

    private PrincipalView assemblePrincipal(final String principal) {
        PrincipalView principalView = new PrincipalView();
        principalView.setFullname(userInfoResolver.getFullNameOf(principal));
        principalView.setUsername(principal);
        return principalView;
    }

    private Map<String, RolePrincipals> principalsByRole(List<RolePrincipals> rolePrincipals) {
        return rolePrincipals.stream().collect(toMap(rp -> rp.getRole().getName(), identity()));
    }

    private Map<String, RolePermissions> permissionsByRole(List<RolePermissions> rolePermissions) {
        return rolePermissions.stream().collect(toMap(rp -> rp.getRole().getName(), identity()));
    }

    private Map<String, Role> rolesByName(List<Role> roles) {
        return roles.stream().collect(toMap(Role::getName, identity()));
    }

    private void checkPermissions(final List<RoleView> roleViewList) {
        Set<String> unknownPermissions = new TreeSet<>();
        roleViewList.forEach(roleView -> unknownPermissions.addAll(roleView.getPermissions().stream().filter(p -> Permission.find(p) == null).collect(toList())));
        checkArgument(unknownPermissions.isEmpty(), "Unknown permissions found: '%s'", unknownPermissions.stream().collect(joining(", ")));
    }

    private void invalidateCachesIfNecessary() {
        try {
            cacheManager.ifPresent(manager -> manager.getCacheNames().forEach(cache -> manager.getCache(cache).invalidate()));
        } catch (Exception e) {
            logger.warn("Unable to clear security cache before operation", e);
        }
    }

}
