package com.xebialabs.xlrelease.actors

import akka.actor.{ActorSystem, CoordinatedShutdown}
import com.typesafe.config.{Config, ConfigFactory}
import com.xebialabs.xlplatform.cluster.NodeState
import com.xebialabs.xlplatform.cluster.full.downing.AutoDowning
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.db.sql.DatabaseInfo
import com.xebialabs.xlrelease.db.sql.DatabaseInfo._
import com.xebialabs.xlrelease.support.akka.GracefulShutdownReason
import com.xebialabs.xlrelease.support.akka.spring.{ScalaSpringAwareBean, SpringExtension}
import grizzled.slf4j.Logging
import org.springframework.beans.factory.FactoryBean
import org.springframework.beans.factory.annotation.Qualifier
import slick.jdbc._

import scala.collection.mutable
import scala.concurrent.Await
import scala.concurrent.duration.Duration
import scala.jdk.CollectionConverters._

class ActorSystemHolder(xlrConfig: XlrConfig, @Qualifier("xlrDatabaseInformation") dbInfo: DatabaseInfo)
  extends Logging with ScalaSpringAwareBean with FactoryBean[ActorSystem] {

  private var running: Boolean = false

  lazy val actorSystem: ActorSystem = {
    info("Starting up actor system.")
    if (xlrConfig.isActiveOnStartup) {
      NodeState.setActive(true)
    }

    val akkaConfig = getSlickConfig.withFallback(xlrConfig.akka.config)
    val system = ActorSystem(xlrConfig.akka.systemName, akkaConfig)
    system.registerExtension(SpringExtension)
    running = true
    system
  }

  def shutdown(): Unit = {
    if (running) {
      info("Initiating actor system shutdown.")
      if (!AutoDowning.isDowning) {
        // Actor system should be terminated by Coordinated Shutdown Akka extension
        val result = CoordinatedShutdown.get(actorSystem).run(GracefulShutdownReason)
        info("Waiting for actor system shutdown (indefinitely).")
        Await.result(result, atMost = Duration.Inf)
      }
      running = false
      info("Actor system shutdown finished.")
    }
  }

  def getInstance(): ActorSystem = actorSystem

  //get slick config for akka-persistence-jdbc
  private def getSlickConfig: Config = {
    val slickProfileConfigKey: String = "akka-persistence-jdbc.shared-databases.slick.profile"
    val configMap: mutable.Map[String, String] = mutable.Map()

    dbInfo match {
      case H2(_) =>
        configMap.put(slickProfileConfigKey, H2Profile.getClass.getName)
      case Derby(_) =>
        configMap.put(slickProfileConfigKey, DerbyProfile.getClass.getName)
      case MsSqlServer(_) =>
        configMap.put(slickProfileConfigKey, SQLServerProfile.getClass.getName)
      case MySql(_) =>
        configMap.put(slickProfileConfigKey, MySQLProfile.getClass.getName)
      case Oracle(_) =>
        configMap.put(slickProfileConfigKey, OracleProfile.getClass.getName)
      case PostgreSql(_) =>
        configMap.put(slickProfileConfigKey, PostgresProfile.getClass.getName)

        //for PostgreSql, we have to set table names and column names in lower case
        configMap.put("jdbc-journal.tables.legacy_journal.tableName", "a_job_journal")
        configMap.addAll(getJournalTableColumnConfig("jdbc-journal"))

        configMap.put("jdbc-snapshot-store.tables.legacy_snapshot.tableName", "a_job_snapshot")
        configMap.addAll(getSnapshotTableColumnConfig("jdbc-snapshot-store"))

        configMap.put("sharding-jdbc-journal.tables.legacy_journal.tableName", "a_shard_journal")
        configMap.addAll(getJournalTableColumnConfig("sharding-jdbc-journal"))

        configMap.put("sharding-jdbc-snapshot-store.tables.legacy_snapshot.tableName", "a_shard_snapshot")
        configMap.addAll(getSnapshotTableColumnConfig("sharding-jdbc-snapshot-store"))
      case TestDatabaseInfo(_) =>
        configMap.put(slickProfileConfigKey, H2Profile.getClass.getName)
      case Db2(_) =>
        logger.warn("DB2 Database does not support container based tasks.")
        configMap.put(slickProfileConfigKey, H2Profile.getClass.getName)
      case _ => throw new IllegalStateException("Unsupported database detected")
    }

    ConfigFactory.parseMap(configMap.asJava)
  }

  private def getJournalTableColumnConfig(journalName: String): Map[String, String] = {
    val columnNameConfigKey = s"$journalName.tables.legacy_journal.columnNames"
    Map (
      s"$columnNameConfigKey.ordering" -> "ordering",
      s"$columnNameConfigKey.deleted" -> "deleted",
      s"$columnNameConfigKey.persistenceId" -> "persistence_id",
      s"$columnNameConfigKey.sequenceNumber" -> "sequence_number",
      s"$columnNameConfigKey.created" -> "created",
      s"$columnNameConfigKey.tags" -> "tags",
      s"$columnNameConfigKey.message" -> "message"
    )
  }

  private def getSnapshotTableColumnConfig(snapshotName: String): Map[String, String] = {
    val columnNameConfigKey = s"$snapshotName.tables.legacy_snapshot.columnNames"
    Map (
      s"$columnNameConfigKey.persistenceId" -> "persistence_id",
      s"$columnNameConfigKey.sequenceNumber" -> "sequence_number",
      s"$columnNameConfigKey.created" -> "created",
      s"$columnNameConfigKey.snapshot" -> "snapshot"
    )
  }

  override def getObject: ActorSystem = actorSystem

  override def getObjectType: Class[_] = classOf[ActorSystem]
}
