package com.xebialabs.xlrelease.repository;

import java.util.*;
import javax.xml.bind.DatatypeConverter;

import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.PropertyKind;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.CiAttributes;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.ExternalProperty;
import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItem;
import com.xebialabs.deployit.plugin.api.validation.ValidationMessage;
import com.xebialabs.xlrelease.utils.CiHelper;
import com.xebialabs.xltype.serialization.CiListReader;
import com.xebialabs.xltype.serialization.CiReader;

import static java.util.Collections.emptyList;

public class CiMemoryReader implements CiReader {

    protected final ConfigurationItem ci;
    private final Iterator<PropertyDescriptor> propertyIterator;
    private final Type targetType;
    private PropertyDescriptor currentProperty;

    public CiMemoryReader(final ConfigurationItem ci) {
        this.ci = ci;
        this.targetType = this.ci.getType();
        propertyIterator = targetType.getDescriptor().getPropertyDescriptors().iterator();
    }

    public CiMemoryReader(final ConfigurationItem ci, final Type targetType) {
        this.ci = ci;
        this.targetType = targetType;
        propertyIterator = targetType.getDescriptor().getPropertyDescriptors().stream()
                .filter(pd -> ci.hasProperty(pd.getName()))
                .iterator();
    }

    @Override
    public String getType() {
        return targetType.toString();
    }

    @Override
    public String getId() {
        return ci.getId();
    }

    @Override
    public String getToken() {
        if (ci instanceof BaseConfigurationItem) {
            return ((BaseConfigurationItem) ci).get$token();
        }
        return null;
    }

    @Override
    public CiAttributes getCiAttributes() {
        if (ci instanceof BaseConfigurationItem) {
            return ((BaseConfigurationItem) ci).get$ciAttributes();
        }
        return null;
    }

    @Override
    public boolean hasMoreProperties() {
        return propertyIterator.hasNext();
    }

    @Override
    public void moveIntoProperty() {
        currentProperty = propertyIterator.next();
    }

    @Override
    public CiReader moveIntoNestedProperty() {
        ConfigurationItem nestedCi = (ConfigurationItem) currentProperty.get(this.ci);
        return new CiMemoryReader(nestedCi);
    }

    @Override
    public void moveOutOfProperty() {
    }

    @Override
    public String getCurrentPropertyName() {
        return currentProperty.getName();
    }

    @Override
    public String getStringValue() {
        Object value = currentProperty.get(this.ci);
        if (value == null) {
            return null;
        }
        if (value instanceof Date) {
            Calendar calendar = GregorianCalendar.getInstance();
            calendar.setTime((Date) value);
            return DatatypeConverter.printDateTime(calendar);
        }
        if (value instanceof ConfigurationItem) {
            return ((ConfigurationItem) value).getId();
        }
        return String.valueOf(value);
    }

    @Override
    public List<String> getStringValues() {
        @SuppressWarnings("unchecked")
        Collection<String> values = (Collection<String>) currentProperty.get(this.ci);
        return values == null ? null : new ArrayList<>(values);
    }

    @Override
    public Map<String, String> getStringMap() {
        @SuppressWarnings("unchecked")
        Map<String, String> values = (Map<String, String>) currentProperty.get(this.ci);
        return values == null ? null : new LinkedHashMap<>(values);
    }

    @Override
    public boolean isCiReference() {
        if (currentProperty == null || currentProperty.getKind() != PropertyKind.CI) {
            return false;
        }
        ConfigurationItem value = (ConfigurationItem) currentProperty.get(this.ci);
        return value == null || !CiHelper.isChildViaOneOfChildProperties(value, this.ci);
    }

    @Override
    public String getCiReference() {
        return getStringValue();
    }

    @Override
    public List<String> getCiReferences() {
        return getStringValues();
    }

    @Override
    public CiListReader getCurrentCiListReader() {
        @SuppressWarnings("unchecked")
        Collection<ConfigurationItem> referencedCis = (Collection<ConfigurationItem>) currentProperty.get(this.ci);
        if (currentProperty.isAsContainment()) {
            return new CiListMemoryReader(referencedCis);
        } else {
            return new CiListAsReferencesReader(referencedCis);
        }
    }

    @Override
    public List<ValidationMessage> getValidationMessages() {
        return emptyList();
    }

    @Override
    public Map<String, ExternalProperty> getExternalProperties() {
        return new HashMap<>();
    }
}
