package com.xebialabs.xlrelease.scheduler.logs

import akka.actor.{Actor, ActorLogging, ActorRef}
import akka.event.LoggingReceive
import com.xebialabs.xlrelease.scheduler.logs.ExecutionLogWatchActor.{StartWatch, StopWatching, WatcherMsg}
import com.xebialabs.xlrelease.support.akka.spring.{SpringActor, SpringExtension}

@SpringActor
class ExecutionLogWatcherSupervisor extends Actor with ActorLogging {
  override def receive: Receive = LoggingReceive {
    handleMsg(Set())
  }

  private def handleMsg(watchedExecutionIds: Set[String]): Receive = {
    case m: WatcherMsg => handleWatcherMsg(watchedExecutionIds)(m)
    case m => log.error(s"Ignoring unexpected message $m")
  }

  private def handleWatcherMsg(watchedExecutionIds: Set[String] = Set())(m: WatcherMsg): Unit = m match {
    case m@StartWatch(_, executionId, _) =>
      // in theory there is a chance that actor will be gone at this point while we still have actorRef
      //  which means that this message will end in dead letter queue and we actor will not be started
      //  but it is not worth proper fix because it's a read only operation and it can be retried
      logWatchActor(executionId) ! m
      context.become(handleMsg(watchedExecutionIds = watchedExecutionIds + m.executionId))
    case m@StopWatching(executionId, _) => if (watchedExecutionIds.contains(m.executionId)) {
      logWatchActor(executionId) ! m
      context.become(handleMsg(watchedExecutionIds = watchedExecutionIds - executionId))
    }
    case m => if (watchedExecutionIds.contains(m.executionId)) {
      logWatchActor(m.executionId) ! m
    }
  }

  private def logWatchActor(executionId: String): ActorRef = {
    val actorName = s"execution-log-watcher-${executionId}"
    // for now we do not want to create dedicated dispatcher for these actors - we will use default one
    val actorRef = context.child(actorName).getOrElse {
      SpringExtension.childActorOf(classOf[ExecutionLogWatchActor], actorName)(context)
    }
    context.watchWith(actorRef, StopWatching(executionId, None))
    actorRef
  }

}
