package com.xebialabs.xlrelease.actors

import com.esotericsoftware.kryo.Kryo
import com.esotericsoftware.kryo.util.DefaultClassResolver
import com.xebialabs.xlplatform.utils.ClassLoaderUtils.classLoader
import com.xebialabs.xlrelease.actors.kryoserializers._
import com.xebialabs.xlrelease.repository.proxy.{ResolvableConfigurationItemReference, ResolvableLazyConfigurationItem}
import com.xebialabs.xlrelease.script.TaskSoftReference
import grizzled.slf4j.Logging
import io.altoo.serialization.kryo.pekko.DefaultKryoInitializer
import io.altoo.serialization.kryo.scala.serializer._

class KryoInit extends DefaultKryoInitializer with Logging {
  override def postInit(kryo: ScalaKryo): Unit = {
    JavaKaffeeSerializers(kryo)
    TwitterChillSerializers(kryo)

    // kryo.setWarnUnregisteredClasses(false)
    kryo.addDefaultSerializer(classOf[ResolvableLazyConfigurationItem], ResolvableLazyConfigurationItemSerializer)
    kryo.addDefaultSerializer(classOf[TaskSoftReference[_]], TaskSoftReferenceSerializer)
    kryo.register(classOf[ResolvableLazyConfigurationItem])
    kryo.register(classOf[ResolvableConfigurationItemReference])

    // mechanism so we can register custom serializers from other spring modules
    KryoInitializers.postInit(kryo)

    val tccl = classLoader
    kryo.setClassLoader(tccl)

    val classResolver = kryo.getClassResolver
    classResolver match {
      case resolver: DefaultClassResolver =>
         // I really did not want to use reflection, but alternative is to re-implement/copy KryoSerializer (from altoo) in order to use custom classresolver
         replaceClassResolver(kryo, resolver)
      case _ =>
        throw new IllegalStateException(s"Unable to replace unknown classResolver type ${classResolver.getClass.getName}")
    }
  }

  private def replaceClassResolver(kryo: ScalaKryo, original: DefaultClassResolver): Unit = {
    // Kryo might use CollectionSerializer under the hood to serialize "syntheticProperties" collection.
    // If this happens it will write name of the class and then serialize it using registered serializer
    // If collection element is a cglib proxy it means that the class was generated only on the node where serialization happens
    // This is a problem because other nodes in the cluster do not know about this custom class - those nodes do have serializer
    // and can deserialize a reference into their own proxy.
    // That's why we want to tell Kryo that it should use "ResolvableLazyConfigurationItem" as a class name instead of the one created by cglib.
    // Use of KryoSerializer and Pekko make this difficult to test on a single node and with cluster you have to debug through Kryo codebase
    // to see which serializers and class names are being picked up.
    val kryoClazz = classOf[Kryo]
    val classResolverField = kryoClazz.getDeclaredField("classResolver")
    classResolverField.setAccessible(true)
    val xlrClassResolver = new XlrClassResolver(original)
    classResolverField.set(kryo, xlrClassResolver)
    xlrClassResolver.setKryo(kryo)
  }

}
